package ru.nilsoft.example;

import android.content.DialogInterface;
import android.os.Handler;
import androidx.appcompat.app.AppCompatActivity;
import android.os.Bundle;
import android.view.View;
import android.widget.TextView;

import java.util.Calendar;

import ru.nilsoft.tm.TMCommand;
import ru.nilsoft.tm.TMError;
import ru.nilsoft.tm.TMLib;
import ru.nilsoft.tm.TMLibHandler;

/**
 * Настройка/установка времени контроллера ФН.
 * - синхронизация КФН по времени Android;
 * - изменение времени в контроллере ФН.
 *
 * Внимание: UTC зону контроллера ФН можно сохранить только из приложения ККТ (так как она хранится в настройках сервиса).
 *
 * @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class SetTimeActivity extends AppCompatActivity implements MessageBox.CallBack{
    /** Handler для работы c контроллером ФН. */
    private Handler h;
    /** Выход из активности. */
    private boolean isExit = false;
    /** Данные для установки даты. */
    private String setDate = "";
    /** Данные для установки времени. */
    private String setTime = "";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_set_time);

        //создание хендлера
        h = new LibHandler();
    }

    @Override
    protected void onStart() {
        super.onStart();

        //регистрация хендлера для получения ответов от контроллера ФН
        TMLib.getInstance().registerHandler(h);

        //получения текущего времени
        FRCmdGetDateTime();
    }

    @Override
    protected void onStop() {
        //освобождение хендлера
        TMLib.getInstance().unregisterHandler(h);

        super.onStop();
    }

    /** При позитивном завершении диалога (нажатие кнопки ОК). */
    public void onMessageBoxPositiveClick(DialogInterface dialog) {
        //завершаем активность
        if ( isExit ) finish();
    }

    /** При негативном завершении диалога (нажатие кнопки ОТМЕНА). */
    public void onMessageBoxNegativeClick(DialogInterface dialog) {
    }

    /** Обработка нажатия кнопки ОТМЕНА. */
    public void onButtonCancel(View view) {
        finish(); //завершение активности
    }

    /** Обработка нажатия кнопки СОХРАНИТЬ ВРЕМЯ В КОНТРОЛЛЕР ФН. */
    public void onButtonSetTime(View view) {
        setDate = String.valueOf(((TextView)findViewById(R.id.editDate)).getText());
        setTime = String.valueOf(((TextView)findViewById(R.id.editTime)).getText());
        FRCmdSetDateTime(setDate, setTime);
    }

    /** Обработка нажатия кнопки СИНХРОНИЗИРОВАТЬ ПО ВРЕМЕНИ ТЕРМИНАЛА. */
    public void onButtonSync(View view) {
        Calendar cal = Calendar.getInstance();
        setDate = String.format("%02d%02d%02d",cal.get(Calendar.DAY_OF_MONTH),cal.get(Calendar.MONTH)+1,cal.get(Calendar.YEAR)%100);
        setTime = String.format("%02d%02d",cal.get(Calendar.HOUR_OF_DAY),cal.get(Calendar.MINUTE));
        FRCmdSetDateTime(setDate, setTime);
    }

    /** Выполнение команды получения времени. */
    public void FRCmdGetDateTime() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetDateTime();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /** Выполнение команды установки времени. */
    public void FRCmdSetDateTime(String d, String t) {
        TMCommand cmd = new TMCommand();
        cmd.CmdSetDateTime(d,t);
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /** Хендлер для обработки сообщений от контроллера ФН. */
    private class LibHandler extends TMLibHandler {

        /** Конструктор по умолчанию. */
        public LibHandler(){
            super(MainApp.GetCtx().getMainLooper());
        }

        @Override
        public void onRespCmd(TMCommand cmd) {
            //получен ответ на команду
            if ( cmd.GetMainError() == 0 ){
                //обработка успешных ответов
                switch ((byte) cmd.GetCmdCode() ) {
                    case TMCommand.commands.CM_GetDateTime:
                        //разборка параметров и установка их в поля
                        TextView tDate = findViewById(R.id.editDate);
                        TextView tTime = findViewById(R.id.editTime);
                        tDate.setText(cmd.GetFieldString(5));
                        tTime.setText(cmd.GetFieldString(6));
                        //время терминала
                        TextView curDate = findViewById(R.id.textCurDate);
                        TextView curTime = findViewById(R.id.textCurTime);
                        TextView curUTC = findViewById(R.id.textCurUTC);
                        Calendar cal = Calendar.getInstance();
                        curDate.setText(String.format("%02d%02d%02d",cal.get(Calendar.DAY_OF_MONTH),cal.get(Calendar.MONTH)+1,cal.get(Calendar.YEAR)%100));
                        curTime.setText(String.format("%02d%02d",cal.get(Calendar.HOUR_OF_DAY),cal.get(Calendar.MINUTE)));
                        curUTC.setText(String.format("%d",cal.get(Calendar.ZONE_OFFSET)/(60*60*1000)));
                        //запрос версии сервера (в ответе на этот запрос приходит UTC контроллера ФН)
                        TMLib.getInstance().SendRespVer();
                        break;

                    case TMCommand.commands.CM_SetDateTime:
                        //данные сохранены
                        isExit = true;
                        MessageBox.Create(getString(R.string.header_success), getString(R.string.msg_cmd_success)+"\n\nВРЕМЯ УСТАНОВЛЕНО", MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
                        break;
                }
            }
            else {
                //обработка ошибок
                if ( ( cmd.GetCmdCode() == TMCommand.commands.CM_SetDateTime ) && ( cmd.GetMainError() == TMCommand.errors.ERR_TIME_TOOLARGE ) ) {
                    //повторная отправка установки времени если разница времени слишком большая
                    FRCmdSetDateTime(setDate,setTime);
                }
                else {
                    //сообщение об ошибке выполнения команды
                    MessageBox.Create(getString(R.string.header_error), String.format(getString(R.string.msg_cmd_error), cmd.GetMainError(), cmd.GetSubError(), cmd.GetCmdCode()) + "\n\n" + TMError.GetText((byte) cmd.GetMainError()), MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
                }
            }
        }

        @Override
        public void onVersion(Bundle data, int hi_ver, int mi_ver, int lo_ver, int date_ver, String text) {
            //в ответе на запрос версии возвращается UTC контроллера ФН (зона хранится в настройках сервиса)
            TextView tUTC = findViewById(R.id.editUTC);
            tUTC.setText(String.valueOf(data.getInt("utc",3))); //по умолчанию делаем по МОСКВЕ
        }

        @Override
        public void onTimeOut() {
            //сообщение о превышении таймаута обращения к контроллеру ФН
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_timeout), MessageBox.option.OK|MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }

        @Override
        public void onError(String errText) {
            //сообщение о сбое сервиса взаимодействия с контроллером ФН
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_service_fail)+"\n\n"+errText, MessageBox.option.OK|MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }
    }
}
