package ru.nilsoft.example;

import android.content.DialogInterface;
import android.os.Handler;
import androidx.appcompat.app.AppCompatActivity;
import android.os.Bundle;
import android.view.View;
import android.widget.TextView;

import ru.nilsoft.tm.TMCommand;
import ru.nilsoft.tm.TMError;
import ru.nilsoft.tm.TMLib;
import ru.nilsoft.tm.TMLibHandler;

/**
 * Настройка параметров обмена с серверами ОФД/ОИСМ/АС ОКП.
 *
 * Параметры храняться во внутренней памяти контроллера ФН.
 * Используются следующие параметры:
 * - Адрес ОФД (DNS или IP сервера ОФД);
 * - Порт ОФД;
 * - Таймаут обращения (время повтора обращения к ОФД в случае ошибки связи, используется только при фоновой отправке);
 * - Таймаут ожидания (время ожидания ответа сервера ОФД);
 * - Таймаут отправки (время отправки запроса на сервер ОФД).
 *
 * @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class SetOfdActivity extends AppCompatActivity implements MessageBox.CallBack{
    public static final String SERVER_TYPE = "server_type";
    /** Handler для работы c контроллером ФН. */
    private Handler h;
    /** Версия ФФД. */
    private byte ffdVer = TMCommand.proto_ffd.FFD_1_05;
    /** Флаг TMT. */
    private boolean isTMT = false;
    /** Номер тега. */
    private short tag = 0;
    /** Тип сервера. */
    private byte serverType = TMCommand.server_types.OFD;
    /** Выход из диалога. */
    private int isExit = 0;
    /** Поле ввода адреса сервера. */
    TextView tAddress;
    /** Поле ввода порта сервера. */
    TextView tPort;
    /** Поле ввода времени соединения. */
    TextView tTimeoutConn;
    /** Поле ввода времени ожидания ответа. */
    TextView tTimeoutRecv;
    /** Поле ввода времени отправки сообщения. */
    TextView tTimeoutSend;
    /** Название сервера. */
    String tHeader;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setContentView(R.layout.activity_set_ofd);

        //создание хендлера
        h = new LibHandler();
   }

    @Override
    protected void onStart() {
        super.onStart();

        //регистрация хендлера для получения ответов от контроллера ФН
        TMLib.getInstance().registerHandler(h);

        //получение параметра регистрации ФН (протокол ФФД)
        FRFNGetRegParam((short)1209);
    }

    @Override
    protected void onStop() {
        //освобождение хендлера
        TMLib.getInstance().unregisterHandler(h);

        super.onStop();
    }

    /** При позитивном завершении диалога (нажатие кнопки ОК). */
    public void onMessageBoxPositiveClick(DialogInterface dialog) {
        //завершаем диалог
        if ( isExit != 0 ) finish();
    }

    /** При негативном завершении диалога (нажатие кнопки ОТМЕНА). */
    public void onMessageBoxNegativeClick(DialogInterface dialog) {
    }

    /** Обработка нажатия кнопки OK. */
    public void onButtonOK(View view) {
        //установку начинаем с сервера ОФД
        FRSetOFDSetParam(TMCommand.server_types.OFD);
    }

    /** Обработка нажатия кнопки ОТМЕНА. */
    public void onButtonCancel(View view) {
        finish(); //завершение диалога
    }

    /**
     * Получение значений из полей ввода диалога.
     * @return true: значения считаны.
     */
    public boolean getEdit() {
        switch(this.serverType) {
            case TMCommand.server_types.OFD:
                tAddress = findViewById(R.id.ofd_address);
                tPort = findViewById(R.id.ofd_port);
                tTimeoutConn = findViewById(R.id.ofd_timeout_conn);
                tTimeoutRecv = findViewById(R.id.ofd_timeout_recv);
                tTimeoutSend = findViewById(R.id.ofd_timeout_send);
                tHeader = getString(R.string.header_server_ofd);
                break;

            case TMCommand.server_types.OISM:
                tAddress = findViewById(R.id.oism_address);
                tPort = findViewById(R.id.oism_port);
                tTimeoutConn = findViewById(R.id.oism_timeout_conn);
                tTimeoutRecv = findViewById(R.id.oism_timeout_recv);
                tTimeoutSend = findViewById(R.id.oism_timeout_send);
                tHeader = getString(R.string.header_server_oism);
                break;

            case TMCommand.server_types.ASOKP:
                tAddress = findViewById(R.id.asokp_address);
                tPort = findViewById(R.id.asokp_port);
                tTimeoutConn = findViewById(R.id.asokp_timeout_conn);
                tTimeoutRecv = findViewById(R.id.asokp_timeout_recv);
                tTimeoutSend = findViewById(R.id.asokp_timeout_send);
                tHeader = getString(R.string.header_server_asokp);
                break;

            default:
                return false;
        }
        return true;
    }

    /**
     * Установка параметров в строки редактирования.
     * @param address адрес сервера.
     * @param port порт сервера.
     * @param timeoutConn время соединения с сервером (сек).
     * @param timeoutRecv время ожидания ответа (сек).
     * @param timeoutSend время отправки сообщения (сек).
     */
    public void setEdit(String address, int port, int timeoutConn, int timeoutRecv, int timeoutSend) {
        if ( getEdit() ) {
            //заполнение полей ввода
            tAddress.setText(address);
            tPort.setText(String.valueOf(port));
            tTimeoutConn.setText(String.valueOf(timeoutConn));
            tTimeoutRecv.setText(String.valueOf(timeoutRecv));
            tTimeoutSend.setText(String.valueOf(timeoutSend));
        }
    }

    /**
     * Проверка параметров в строках редактирования.
     * @return true: параметры валидные, false: ошибка проверки параметров
     */
    public boolean checkEdit() {
        if ( getEdit() ) {
            String address = String.valueOf(tAddress.getText());
            int port;
            try {
                port = Integer.parseInt(String.valueOf(tPort.getText()));
            } catch (Exception e) {
                port = 0;
            }
            int timeoutConn;
            try {
                timeoutConn = Integer.parseInt(String.valueOf(tTimeoutConn.getText()));
            } catch (Exception e) {
                timeoutConn = 0;
            }
            int timeoutRecv;
            try {
                timeoutRecv = Integer.parseInt(String.valueOf(tTimeoutRecv.getText()));
            } catch (Exception e) {
                timeoutRecv = 0;
            }
            int timeoutSend;
            try {
                timeoutSend = Integer.parseInt(String.valueOf(tTimeoutSend.getText()));
            } catch (Exception e) {
                timeoutSend = 0;
            }

            //проверка значений
            if (address.length() == 0) {
                MessageBox.Create(getString(R.string.header_warning), tHeader + "\n" + getString(R.string.msg_bad_field_url), MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
            } else if ((port == 0) || (port > 65535)) {
                MessageBox.Create(getString(R.string.header_warning), tHeader + "\n" + getString(R.string.msg_bad_field_port), MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
            } else if ((timeoutConn == 0) || (timeoutRecv == 0) || (timeoutSend == 0)) {
                MessageBox.Create(getString(R.string.header_warning), tHeader + "\n" + getString(R.string.msg_bad_field_timeout_null), MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
            } else if ((timeoutConn > 65535) || (timeoutRecv > 65535) || (timeoutSend > 65535)) {
                MessageBox.Create(getString(R.string.header_warning), tHeader + "\n" + getString(R.string.msg_bad_field_timeout), MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
            } else {
                return true;
            }
        }

        return false;
    }

    /**
     * Отправка в КФН команды установки параметров сервера.
     * @param serverType тип сервера.
     */
    private void FRSetOFDSetParam(byte serverType) {

        this.serverType = serverType;
        if ( checkEdit() ) {
            TMCommand cmd = new TMCommand();

            //подготовка команды
            switch (serverType) {
                case TMCommand.server_types.OISM:
                case TMCommand.server_types.ASOKP:
                    //работа через новый функционал
                    cmd.CmdOFDSetParamExt(String.valueOf(tAddress.getText()),
                            (short)Integer.parseInt(String.valueOf(tPort.getText())),
                            (short)Integer.parseInt(String.valueOf(tTimeoutConn.getText())),
                            (short)Integer.parseInt(String.valueOf(tTimeoutRecv.getText())),
                            (short)Integer.parseInt(String.valueOf(tTimeoutSend.getText())),
                            serverType);
                    break;

                case TMCommand.server_types.OFD:
                    //с ОФД работаем по старому, чтобы была совместимость со старыми прошивками КФН
                    cmd.CmdOFDSetParam(String.valueOf(tAddress.getText()),
                            (short)Integer.parseInt(String.valueOf(tPort.getText())),
                            (short)Integer.parseInt(String.valueOf(tTimeoutConn.getText())),
                            (short)Integer.parseInt(String.valueOf(tTimeoutRecv.getText())),
                            (short)Integer.parseInt(String.valueOf(tTimeoutSend.getText())));
                    break;

                default:
                    return;
            }

            //выполнение
            TMLib.getInstance().DoCmd(this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
        }
    }

    /**
     * Команда получения из КФН параметров сервера.
     * @param serverType
     */
    private void FRCmdOFDGetParam(byte serverType) {
        TMCommand cmd = new TMCommand();
        this.serverType = serverType;

        //подготовка команды
        switch(serverType) {
            case TMCommand.server_types.OISM:
            case TMCommand.server_types.ASOKP:
                //работа через новый функционал
                cmd.CmdOFDGetParamExt(serverType);
                break;

            case TMCommand.server_types.OFD:
                //с ОФД работаем по старому, чтобы была совместимость со старыми прошивками КФН
                cmd.CmdOFDGetParam();
                break;

            default:
                return;
        }

        //выполнение
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /** Получение параметров регистрации ФН. */
    public void FRFNGetRegParam(short tlvId) {
        tag = tlvId;

        TMCommand cmd = new TMCommand();
        cmd.CmdFNGetRegParam(tlvId);
        TMLib.getInstance().DoCmd(this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
    }

    /* Хендлер для обработки сообщений от контроллера ФН. */
    private class LibHandler extends TMLibHandler {

        /** Конструктор по умолчанию. */
        public LibHandler(){
            super(MainApp.GetCtx().getMainLooper());
        }

        @Override
        public void onRespCmd(TMCommand cmd) {
            //получен ответ на команду
            if ( cmd.GetMainError() == 0 ){
                //обработка успешных ответов
                switch ((byte) cmd.GetCmdCode()) {
                    case TMCommand.commands.CM_FNGetRegParam:
                        switch(tag) {
                            case (short)1209:
                                //проверка по какому ФФД регистрирован ФН
                                ffdVer = cmd.GetFieldHexB(5);

                                if (ffdVer >= TMCommand.proto_ffd.FFD_1_2) {
                                    //проверяем флаг TMT
                                    FRFNGetRegParam((short)1290);
                                } else {
                                    //получаем параметры сервера
                                    FRCmdOFDGetParam(TMCommand.server_types.OFD);
                                }
                                break;

                            case (short)1290:
                                //проверка наличия флага TMT
                                isTMT = (cmd.GetFieldHexDW(5) & TMCommand.kkt_mode_reg.TMT) != 0;
                                if ( isTMT ) {
                                    //разблокируем в случае поддержки КМ
                                    try {
                                        findViewById(R.id.oism_header).setEnabled(true);
                                        findViewById(R.id.oism_address).setEnabled(true);
                                        findViewById(R.id.oism_port).setEnabled(true);
                                        findViewById(R.id.oism_timeout_conn).setEnabled(true);
                                        findViewById(R.id.oism_timeout_recv).setEnabled(true);
                                        findViewById(R.id.oism_timeout_send).setEnabled(true);

                                        findViewById(R.id.asokp_header).setEnabled(true);
                                        findViewById(R.id.asokp_address).setEnabled(true);
                                        findViewById(R.id.asokp_port).setEnabled(true);
                                        findViewById(R.id.asokp_timeout_conn).setEnabled(true);
                                        findViewById(R.id.asokp_timeout_recv).setEnabled(true);
                                        findViewById(R.id.asokp_timeout_send).setEnabled(true);
                                    }
                                    catch (Exception ignored) {}
                                }

                                //получаем параметры сервера
                                FRCmdOFDGetParam(TMCommand.server_types.OFD);
                                break;
                        }
                        break;

                    case TMCommand.commands.CM_OFDGetParam:
                        //разборка параметров и установка их в поля
                        setEdit(cmd.GetFieldString(9),
                                0xFFFF&cmd.GetFieldHexW(5),
                                0xFFFF&cmd.GetFieldHexW(6),
                                0xFFFF&cmd.GetFieldHexW(7),
                                0xFFFF&cmd.GetFieldHexW(8));

                        if ( ( ffdVer >= TMCommand.proto_ffd.FFD_1_2 ) && isTMT ) {
                            //получаем параметры следующего типа сервера
                            switch (serverType) {
                                case TMCommand.server_types.OFD:
                                    FRCmdOFDGetParam(TMCommand.server_types.OISM);
                                    break;
                                case TMCommand.server_types.OISM:
                                    FRCmdOFDGetParam(TMCommand.server_types.ASOKP);
                                    break;

                            }
                        }
                        break;

                    case TMCommand.commands.CM_OFDSetParam:
                        if ( ( ffdVer >= TMCommand.proto_ffd.FFD_1_2 ) && isTMT && ( serverType != TMCommand.server_types.ASOKP ) ) {
                            //устанавливаем параметры следующего типа сервера
                            switch (serverType) {
                                case TMCommand.server_types.OFD:
                                    FRSetOFDSetParam(TMCommand.server_types.OISM);
                                    break;
                                case TMCommand.server_types.OISM:
                                    FRSetOFDSetParam(TMCommand.server_types.ASOKP);
                                    break;
                            }
                        }
                        else {
                            isExit = 1;
                            MessageBox.Create(getString(R.string.header_success), getString(R.string.msg_cmd_success) + "\n\nПАРАМЕТРЫ СОХРАНЕНЫ", MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
                        }
                        break;
                }
            }
            else {
                //обработка ошибок
                MessageBox.Create(getString(R.string.header_error), String.format(getString(R.string.msg_cmd_error),cmd.GetMainError(),cmd.GetSubError(),cmd.GetCmdCode())+"\n\n"+TMError.GetText((byte)cmd.GetMainError()), MessageBox.option.OK|MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
            }
        }

        @Override
        public void onTimeOut() {
            //сообщение о превышении таймаута обращения к контроллеру ФН
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_timeout), MessageBox.option.OK|MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }

        @Override
        public void onError(String errText) {
            //сообщение о сбое сервиса взаимодействия с контроллером ФН
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_service_fail)+"\n\n"+errText, MessageBox.option.OK|MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }
    }
}
