package ru.nilsoft.example;

import android.content.Context;
import android.content.DialogInterface;
import android.content.SharedPreferences;

import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import android.os.Bundle;
import android.view.View;
import android.widget.TextView;

import ru.nilsoft.tm.TMCommand;
import ru.nilsoft.tm.TMError;
import ru.nilsoft.tm.TMKeysUpdate;
import ru.nilsoft.tm.TMLib;

/**
 * Открытие смены:
 * - ввод ФИО и ИНН кассира;
 * - проверка необходимости обновления ключей;
 * - открытие смены на ККТ.
 *
 * @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class OpenShiftActivity extends AppCompatActivity implements MessageBox.CallBack{
    /** Handler для работы c TMLib. */
    private TMKeysUpdate h;
    /** Для однократного старта. */
    private boolean isStarted = false;
    /** Выход из диалога. */
    private boolean isExit = false;
    /** Диалог для вывода сообщения. */
    private AlertDialog dialog = null;
    /** Переменная устанавливается в true если диалог надо закрыть. */
    private boolean stopDlg = false;
    /** Есть ли работа с УПД. */
    private boolean isUPD = false;


    /** Названия настроек для сохранения в файл настроек приложения */
    public static final class sets{
        /** Имя пользователя. */
        static final String USER_NAME = "OpenShift_UserName";
        /** ИНН пользователя. */
        static final String USER_INN = "OpenShift_UserINN";
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_open_shift);

        //создание хендлера
        h = new LibHandler();
    }

    @Override
    protected void onStart() {
        super.onStart();

        //регистрация хендлера для получения ответов от контроллера ФН
        TMLib.getInstance().registerHandler(h);

        if ( !isStarted ) {
            isStarted = true;

            //заполнение данными (считываются из файла настроек приложения)
            ((TextView) findViewById(R.id.fio_kassir)).setText(MainApp.settings.getString(sets.USER_NAME, ""));
            ((TextView) findViewById(R.id.inn_kassir)).setText(MainApp.settings.getString(sets.USER_INN, ""));
        }

        //завершение диалога
        if ( stopDlg ) stopDialog();
    }

    @Override
    protected void onStop() {
        //освобождение хендлера
        TMLib.getInstance().unregisterHandler(h);

        //сохранение данных (сохранение в файл настроек приложения)
        SharedPreferences.Editor editor = MainApp.settings.edit();
        editor.putString(sets.USER_NAME,String.valueOf(((TextView)findViewById(R.id.fio_kassir)).getText()));
        editor.putString(sets.USER_INN,String.valueOf(((TextView)findViewById(R.id.inn_kassir)).getText()));
        editor.apply();

        super.onStop();
    }

    /** При позитивном завершении диалога (нажатие кнопки ОК). */
    public void onMessageBoxPositiveClick(DialogInterface dialog) {
        //завершаем диалог
        if ( isExit ) finish();
    }

    /** При негативном завершении диалога (нажатие кнопки ОТМЕНА). */
    public void onMessageBoxNegativeClick(DialogInterface dialog) {
    }

    /** Обработка нажатия кнопки OK. */
    public void onButtonOK(View view) {
        //проверка введенных значений
        if ( String.valueOf(((TextView)findViewById(R.id.fio_kassir)).getText()).length() == 0 ) {
            MessageBox.Create(getString(R.string.header_warning), getString(R.string.msg_bad_field_cashier), MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
        }
        else {
            //создаем и выводим блокирующий диалог
            AlertDialog.Builder builder = new AlertDialog.Builder(this);
            builder.setTitle(R.string.header_wait);
            builder.setMessage(R.string.msg_update_keys);
            dialog = builder.create(); //установка текста
            dialog.setCancelable(false); //игнорирование клавиши возврат
            dialog.setCanceledOnTouchOutside(false); //игнорирование тычков за пределами диалога
            dialog.show();

            //создаем процедуры обратного вызова для процедуры обновления ключей
            TMKeysUpdate.CallBack cb = new TMKeysUpdate.CallBack() {
                @Override
                public void onKeysCheckError(String errText) {
                    //если ошибка проверки необходимости обновления ключей, запускаем открытие смены
                    startOpenShift();
                }

                @Override
                public void onKeysUpdateError(int flag, String errText) {
                    //при ошибке обновления ключей:
                    //подготовка сообщения об ошибке обновлении ключей для печати на чеке открытия смены
                        MessageBox.CallBack mc = new MessageBox.CallBack() {
                            @Override
                            public void onMessageBoxPositiveClick(DialogInterface dialogInterface) {
                                //запускаем открытие смены ККТ
                                startOpenShift();
                            }

                            @Override
                            public void onMessageBoxNegativeClick(DialogInterface dialogInterface) {
                                //запускаем открытие смены ККТ
                                startOpenShift();
                            }
                        };
                        MessageBox.Create(getString(R.string.header_error),  errText, MessageBox.option.OK | MessageBox.option.WARN, mc).show(getSupportFragmentManager(), MessageBox.TAG);
                }

                @Override
                public void onKeysUpdateSuccess() {
                    //при успешном обновлении ключей:
                    //подготовка сообщения об успешном обновлении ключей для печати на чеке открытия смены
                    //запускаем открытие смены на ККТ
                    startOpenShift();
                }

                @Override
                public void onKeysUpdateCancel() {
                    //если процедура обновления ключей не требуется:
                    //запускаем открытие смены на ККТ
                    startOpenShift();
                }
            };

            //запуск проверки необходимости обновления ключей
            h.StartCheck(null,cb);
        }
    }

    /** Обработка нажатия кнопки ОТМЕНА. */
    public void onButtonCancel(View view) {
        finish(); //завершение активности
    }

    /** Останов диалога об обновлении ключей. */
    private void stopDialog() {
        runOnUiThread(() -> {
            if ( dialog != null ) {
                try{
                    stopDlg = true;
                    dialog.dismiss();
                    dialog = null;
                }
                catch(Exception ignored) {}
            }
        });
    }

    /** Запуск процедуры открытия смены. */
    private void startOpenShift() {
        stopDialog();

        //начинаем процедуру открытия смены на ККТ:
        //сперва получаем время из контроллера (рекомендуется использовать время ККТ, так как Android время может измениться при пересечении поясов или синхронизации по сети).
        DoCmdGetDateTime();
    }

    /** Выполнение команды получения времени. */
    private void DoCmdGetDateTime() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetDateTime();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Выполнение команды открытия смены.
     * @param tDate текущая дата.
     * @param tTime текущее время.
     * @param operName ФИО кассира.
     * @param operINN ИНН кассира.
     * @param rekvisit дополнительный реквизит.
     */
    private void DoCmdShiftOpen(String tDate, String tTime, String operName, String operINN, String rekvisit) {
        TMCommand cmd = new TMCommand();

        //подготовка команды
        if ( ( tDate != null ) && ( tTime != null ) ) cmd.CmdShiftOpen(tDate, tTime, operName, operINN, null, null, rekvisit);
        else cmd.CmdShiftOpen(operName, operINN, null, null, rekvisit);

        //отправка команды контроллеру ФН
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_PRINT)+((isUPD)?/* обновление ключей через УПД */CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_NETWORK):0));
    }

    /** Хендлер для обработки сообщений от контроллера ФН. */
    private class LibHandler extends TMKeysUpdate {

        /** Конструктор по умолчанию. */
        public LibHandler(){
            super(MainApp.GetCtx().getMainLooper());
        }

        /**
         * Вывод ошибок выполнения команды на контроллере ФН.
         * @param cmd ответ на команду.
         */
        private void viewError(TMCommand cmd) {
            MessageBox.Create(getString(R.string.header_error), String.format(getString(R.string.msg_cmd_error), cmd.GetMainError(), cmd.GetSubError(), cmd.GetCmdCode()) + "\n\n" + TMError.GetText((byte) cmd.GetMainError()), MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }

        @Override
        public void onRespCmd(TMCommand cmd) {
            //получен ответ на команду
            if ( cmd.GetMainError() != 0 )  {
                //сообщение об ошибке выполнения команды
                viewError(cmd);
            }
            else {
                switch((byte) cmd.GetCmdCode()) {
                    case TMCommand.commands.CM_GetDateTime:
                        //если есть УПД
                        if ( (cmd.GetDynamicStatus()&(TMCommand.dynamic_status_flags.UPD_ON|TMCommand.dynamic_status_flags.UPD_PRESENT)) == (TMCommand.dynamic_status_flags.UPD_ON|TMCommand.dynamic_status_flags.UPD_PRESENT) ) isUPD = true;

                        //время получено, теперь выпполняем команду открытия смены на ККТ
                        DoCmdShiftOpen(
                                cmd.GetFieldValue(5),
                                cmd.GetFieldValue(6),
                                String.valueOf(((TextView)findViewById(R.id.fio_kassir)).getText()),
                                String.valueOf(((TextView)findViewById(R.id.inn_kassir)).getText()),
                                String.valueOf(((TextView)findViewById(R.id.tag_1191)).getText()));
                        break;

                    case TMCommand.commands.CM_ShiftOpen:
                        //команда закрытия смены успешно выполнена
                        isExit = true;
                        //отправляем документ в ОФД (запускаем поток фоновой отправки, если он не запущен)
                        TMLib.getInstance().StartOFD();
                        //вывод диалога, что открытие смены на ККТ прошло успешно
                        MessageBox.Create(getString(R.string.header_success), getString(R.string.msg_cmd_success)+"\n\nЗАБЕРИТЕ ЧЕК", MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
                        break;
                }
            }
        }

        @Override
        public void onTimeOut() {
            //сообщение о превышении таймаута к контроллеру ФН
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_timeout), MessageBox.option.OK|MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }

        @Override
        public void onError(String errText) {
            //сообщение о сбое сервиса взаимодействия с контроллером ФН
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_service_fail)+"\n\n"+errText, MessageBox.option.OK|MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }
    }
}
