package ru.nilsoft.example;

import androidx.appcompat.app.AppCompatActivity;
import android.os.Bundle;
import android.text.method.ScrollingMovementMethod;
import android.util.Log;
import android.view.View;
import android.widget.TextView;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;

import ru.nilsoft.tm.TMCommand;
import ru.nilsoft.tm.TMError;
import ru.nilsoft.tm.TMLib;
import ru.nilsoft.tm.TMLibHandler;
import ru.nilsoft.tm.TMOfd;
import ru.nilsoft.tm.TMOism;
import ru.nilsoft.tm.TMTag;

/**
 * Отправка документов в ОФД:
 * - получение количества неотправленных документов;
 * - чтение документа из контроллера ФН;
 * - отправка документа на сервер;
 * - сохранение полученной квитанции в ФН.
 *
 * Для ФФД 1.2 и выше с ТМТ отправка КМ в ОИСМ:
 * - получение количества не отправленых КМ;
 * - чтение КМ из ФН;
 * - отправка КМ на сервер;
 * - сохранение подтверждения.
 *
 * @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class OfdActivity extends AppCompatActivity {
    /** Handler для работы c контроллером ФН. */
    private OfdHandler h;
    /** Для отображения текста. */
    private TextView tView;
    /** Для однократного старта. */
    private boolean isStarted = false;

    /** Класс для отправки на сервер ОФД. */
    private SendOfdPacket sendOfdPack = null;
    /** Пакет для отправки на сервер ОФД. */
    private TMOfd.DataBuf dbOfd = null;

    /** Класс для отправки на сервер ОИСМ. */
    private SendOismPacket sendOismPack = null;
    /** Пакет для отправки на сервер ОИСМ. */
    private TMOism.DataBuf dbOism = null;

    /** Количество неотправленных документов. */
    private int waitDocs = 0;
    /** Серийный номер ФН. */
    private String fnSN = null;
    /** Версия ФФД при регистрации ФН. */
    private byte ffdFN = 0;
    /** Последний тег для команды получения параметров регистрации. */
    protected short regTLV = 0;
    /** Размер данных для отправки. */
    private int packetSize = 0;
    /** Текущиий режим работы {@link state}. */
    private int mode = state.PREPARE;
    /** Статус соединения. */
    private boolean connStatus = false;
    /** Выход из диалога. */
    private boolean isExit = false;

    /** Адрес сервера. */
    private String serverAddress = null;
    /** Порт сервера. */
    private int serverPort = 0;
    /** Время соединения с сервером (сек). */
    private int serverConn = 0;
    /** Время ожидания ответа от сервера (сек). */
    private int serverRecv = 0;
    /** Время отправки пакета на сервер (сек). */
    private int serverSend = 0;

    /** Режимы работы. */
    public static class state {
        /** Подготовка к работе c ОФД. */
        static final int PREPARE = 0;
        /** Чтение документа из ФН. */
        static final int READ    = 1;
        /** Отправка документа в ОФД. */
        static final int SEND    = 2;
        /** Запись документа в ФН. */
        static final int WRITE   = 3;
        /** Сбой при работе. */
        static final int FAIL    = 4;
        /** Прервано при выходе из диалога. */
        static final int ABORT   = 5;
        /** Подготовка к работе c ОИСМ. */
        static final int PREPARE_OISM = 10;
    }

    /** Класс контролирующий обмен с сервером ОФД. */
    private class SendOfdPacket extends Thread {
        /** Признак отмены работы потока. */
        private volatile boolean isCancel = false;

        /** Остановить работу потока. */
        void cancel() {
            isCancel = true;
        }

        /**
         * Получить состояние потока.
         * @return true - поток остановлен снаружи.
         */
        public boolean isCancelled() {
            return isCancel;
        }

        /** Обновление сообщения. */
        void publishProgress(final String text) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    //выполнение в основном потоке приложения
                    addTextToView(text);//вывод сообщения
                }
            });
        }

        /** Действия перед остановом потока. */
        void postExecute() {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    //выполнение в основном потоке приложения
                    if ( mode == state.WRITE ) h.FRFNWriteScript(); //сохраняем пакет в ФН, если получен ответ из ОФД
                    sendOfdPack = null;
                }
            });
        }

        /** Метод работающий в потоке. */
        public void run() {
            String textFail = "";
            SocketChannel s = null;
            InetSocketAddress ip = null;

            //получение адреса хоста
            if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "InetSocketAddress("+serverAddress+","+ serverPort +")");
            try {
                ip = new InetSocketAddress(InetAddress.getByName(serverAddress),serverPort);
                //ip = new InetSocketAddress(ofdAddress,ofdPort);
            }
            catch(Exception e) {
                //ошибка получения адреса хоста
                textFail = "ОШИБКА: При получении адреса. "+e.getMessage();
                mode = state.FAIL;
            }

            //создание сокета
            if ( mode == state.SEND ) {
                if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "open socket");
                try {
                    s = SocketChannel.open();
                    s.configureBlocking(false); //неблокирующий режим для сокета
                }
                catch (Exception e) {
                    //ошибка создания сокета
                    textFail = "ОШИБКА: При открытии сокета. " + e.getMessage();
                    mode = state.FAIL;
                }
            }

            //соединение с ОФД
            if ( mode == state.SEND ) {
                if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "start connect to "+ip.toString());
                try{
                    s.connect(ip);
                    long start = System.currentTimeMillis();
                    do {
                        if ( s.finishConnect() ) break;
                        if ( isCancelled() ) {
                            //прервано пользователем
                            mode = state.ABORT;
                            break;
                        }
                        Thread.sleep(200);
                    }
                    while(System.currentTimeMillis()-start < serverRecv*1000);
                    //анализ состояния
                    if ( mode == state.SEND ) {
                        if (!s.finishConnect()) {
                            //превышен таймаут
                            textFail = "ОШИБКА: Превышено время ожидания соединения с сервером ОФД.";
                            mode = state.FAIL;
                        }
                    }
                }
                catch(InterruptedException ignored) {} //поток прерван - ничего не делаем
                catch(Exception e){
                    //ошибка соeдинения
                    textFail = "ОШИБКА: При соединении. " + e.getMessage();
                    mode = state.FAIL;
                    e.printStackTrace();
                }
            }

            //отправка данных в ОФД
            if ( mode == state.SEND ) {
                if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "start write data");
                try {
                    //подготовка данных к отправке
                    byte [] data = dbOfd.getWithHeader();
                    ByteBuffer buf = ByteBuffer.allocate(data.length);
                    buf.clear();
                    buf.put(data);
                    buf.flip();
                    //отправка
                    long start = System.currentTimeMillis();
                    do {
                        s.write(buf);
                        if( !buf.hasRemaining() ) break; //данные отправлены
                        if ( isCancelled() ) {
                            //прервано пользователем
                            mode = state.ABORT;
                            break;
                        }
                        Thread.sleep(200);
                    }
                    while( System.currentTimeMillis()-start < serverSend*1000 );
                    //анализ состояния
                    if ( mode == state.SEND ) {
                        if (buf.hasRemaining()) {
                            //превышен таймаут
                            textFail = "ОШИБКА: Превышено время отправки на сервер ОФД.";
                            mode = state.FAIL;
                        }
                    }
                }
                catch(InterruptedException ignored) {} //поток прерван - ничего не делаем
                catch(Exception e) {
                    //ошибка соeдинения
                    textFail = "ОШИБКА: При отправке. " + e.getMessage();
                    mode = state.FAIL;
                }
            }

            //получение квитанции из ОФД
            if ( mode == state.SEND ) {
                if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "start read data");
                try {
                    //подготовка буфера
                    int readBytes;
                    dbOfd.clear();
                    ByteBuffer buf = ByteBuffer.allocate(TMCommand.MAX_OFD_DATA_PART_SIZE);
                    //чтение данных
                    long start = System.currentTimeMillis();
                    do {
                        buf.clear();
                        readBytes = s.read(buf);
                        if ( readBytes > 0 ) {
                            //чтение из буфера
                            byte [] data = new byte[readBytes];
                            buf.flip();
                            buf.get(data);
                            //запись в принятые данные
                            dbOfd.add(data);
                            int len = dbOfd.length();
                            if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "read "+ readBytes +" bytes ("+ len +","+ dbOfd.getRespLength() +")");
                            if ( ( len >= 30 ) && ( len == dbOfd.getRespLength() ) )  break; //блок получен полностью
                        }
                        else {
                            if ( readBytes < 0 ) break; //соединение разорвано со стороны сервера
                            if ( isCancelled() ) {
                                //прервано пользователем
                                mode = state.ABORT;
                                break;
                            }
                            Thread.sleep(200);
                        }
                    }
                    while( System.currentTimeMillis()-start < serverRecv*1000 );
                    //анализ состояния
                    if ( mode == state.SEND ) {
                        if (readBytes < 0) {
                            textFail = "ОШИБКА: Cоединение разорвано со стороны сервера ОФД.";
                            mode = state.FAIL;
                        } else if ((dbOfd.getRespLength() != dbOfd.length()) || (dbOfd.length() == 0)) {
                            textFail = "ОШИБКА: Превышено время ожидания ответа сервера ОФД. " + dbOfd.getRespLength();
                            mode = state.FAIL;
                        }
                    }
                }
                catch(InterruptedException ignored) {} //поток прерван - ничего не делаем
                catch(Exception e) {
                    //ошибка соeдинения
                    textFail = "ОШИБКА: При приеме. " + e.getMessage();
                    mode = state.FAIL;
                }
            }

            //закрытие сокета
            if ( s != null ) {
                try {
                    s.close();
                }
                catch(Exception e) {}
            }

            //завершение работы потока
            switch(mode) {
                case state.FAIL:
                    //вывод информации о сбое
                    publishProgress(textFail);
                    break;

                case state.SEND:
                    //данные получены
                    publishProgress(waitDocs +": Ответ из ОФД получен "+ dbOfd.length() +" байт");
                    mode = state.WRITE;
                    break;
            }

            //освобождение ресурсов
            postExecute();
            s = null;
            ip = null;
        }
    }

    /** Класс контролирующий обмен с сервером ОИСМ. */
    private class SendOismPacket extends Thread {
        /** Признак отмены работы потока. */
        private volatile boolean isCancel = false;

        /** Остановить работу потока. */
        void cancel() {
            isCancel = true;
        }

        /**
         * Получить состояние потока.
         * @return true - поток остановлен снаружи.
         */
        public boolean isCancelled() {
            return isCancel;
        }

        /** Обновление текста. */
        void publishProgress(final String text) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    //выполнение в основном потоке приложения
                    addTextToView(text); //добавление сообщения
                }
            });
        }

        /** Действия перед остановом потока. */
        void postExecute() {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    //выполнение в основном потоке приложения
                    if ( mode == state.WRITE ) h.FRNotifWriteReceipt(); //сохраняем пакет в ФН, если получен ответ из ОИСМ
                    sendOismPack = null;
                }
            });
        }

        /** Метод работающий в потоке. */
        public void run() {
            String textFail = "";
            SocketChannel s = null;
            InetSocketAddress ip = null;

            //получение адреса хоста
            if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "InetSocketAddress("+serverAddress+","+ serverPort +")");
            try {
                ip = new InetSocketAddress(InetAddress.getByName(serverAddress),serverPort);
            }
            catch(Exception e) {
                //ошибка получения адреса хоста
                textFail = "ОШИБКА: При получении адреса. "+e.getMessage();
                mode = state.FAIL;
            }

            //создание сокета
            if ( mode == state.SEND ) {
                if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "open socket");
                try {
                    s = SocketChannel.open();
                    s.configureBlocking(false); //неблокирующий режим для сокета
                }
                catch (Exception e) {
                    //ошибка создания сокета
                    textFail = "ОШИБКА: При открытии сокета. " + e.getMessage();
                    mode = state.FAIL;
                }
            }

            //соединение с ОИСМ
            if ( mode == state.SEND ) {
                if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "start connect to "+ip.toString());
                try{
                    s.connect(ip);
                    long start = System.currentTimeMillis();
                    do {
                        if ( s.finishConnect() ) break;
                        if ( isCancelled() ) {
                            //прервано пользователем
                            mode = state.ABORT;
                            break;
                        }
                        Thread.sleep(200);
                    }
                    while(System.currentTimeMillis()-start < serverRecv*1000);
                    //анализ состояния
                    if ( mode == state.SEND ) {
                        if (!s.finishConnect()) {
                            //превышен таймаут
                            textFail = "ОШИБКА: Превышено время ожидания соединения с сервером ОИСМ.";
                            mode = state.FAIL;
                        }
                    }
                }
                catch(InterruptedException ignored) {} //поток прерван - ничего не делаем
                catch(Exception e){
                    //ошибка соeдинения
                    textFail = "ОШИБКА: При соединении. " + e.getMessage();
                    mode = state.FAIL;
                    e.printStackTrace();
                }
            }

            //отправка данных в ОИСМ
            if ( mode == state.SEND ) {
                if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "start write data");
                try {
                    //подготовка данных к отправке
                    byte [] data = dbOism.getWithHeader();
                    ByteBuffer buf = ByteBuffer.allocate(data.length);
                    buf.clear();
                    buf.put(data);
                    buf.flip();
                    //отправка
                    long start = System.currentTimeMillis();
                    do {
                        s.write(buf);
                        if( !buf.hasRemaining() ) break; //данные отправлены
                        if ( isCancelled() ) {
                            //прервано пользователем
                            mode = state.ABORT;
                            break;
                        }
                        Thread.sleep(200);
                    }
                    while( System.currentTimeMillis()-start < serverSend*1000 );
                    //анализ состояния
                    if ( mode == state.SEND ) {
                        if (buf.hasRemaining()) {
                            //превышен таймаут
                            textFail = "ОШИБКА: Превышено время отправки на сервер ОИСМ.";
                            mode = state.FAIL;
                        }
                    }
                }
                catch(InterruptedException ignored) {} //поток прерван - ничего не делаем
                catch(Exception e) {
                    //ошибка соeдинения
                    textFail = "ОШИБКА: При отправке. " + e.getMessage();
                    mode = state.FAIL;
                }
            }

            //получение квитанции из ОИСМ
            if ( mode == state.SEND ) {
                if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "start read data");
                try {
                    //подготовка буфера
                    int readBytes;
                    dbOism.clear();
                    ByteBuffer buf = ByteBuffer.allocate(TMCommand.MAX_OFD_DATA_PART_SIZE);
                    //чтение данных
                    long start = System.currentTimeMillis();
                    do {
                        buf.clear();
                        readBytes = s.read(buf);
                        if ( readBytes > 0 ) {
                            //чтение из буфера
                            byte [] data = new byte[readBytes];
                            buf.flip();
                            buf.get(data);
                            //запись в принятые данные
                            dbOism.add(data);
                            int len = dbOism.length();
                            if (BuildConfig.DEBUG) Log.d(this.getClass().getSimpleName(), "read "+ readBytes +" bytes ("+ len +","+ dbOism.getRespLength() +")");
                            if ( ( len >= 30 ) && ( len == dbOism.getRespLength() ) )  break; //блок получен полностью
                        }
                        else {
                            if ( readBytes < 0 ) break; //соединение разорвано со стороны сервера
                            if ( isCancelled() ) {
                                //прервано пользователем
                                mode = state.ABORT;
                                break;
                            }
                            Thread.sleep(200);
                        }
                    }
                    while( System.currentTimeMillis()-start < serverRecv*1000 );
                    //анализ состояния
                    if ( mode == state.SEND ) {
                        if (readBytes < 0) {
                            textFail = "ОШИБКА: Cоединение разорвано со стороны сервера ОИСМ.";
                            mode = state.FAIL;
                        } else if ((dbOism.getRespLength() != dbOism.length()) || (dbOism.length() == 0)) {
                            textFail = "ОШИБКА: Превышено время ожидания ответа сервера ОИСМ. " + dbOism.getRespLength();
                            mode = state.FAIL;
                        }
                    }
                }
                catch(InterruptedException ignored) {} //поток прерван - ничего не делаем
                catch(Exception e) {
                    //ошибка соeдинения
                    textFail = "ОШИБКА: При приеме. " + e.getMessage();
                    mode = state.FAIL;
                }
            }

            //закрытие сокета
            if ( s != null ) {
                try {
                    s.close();
                }
                catch(Exception ignored) {}
            }

            //завершение работы потока
            switch(mode) {
                case state.FAIL:
                    //вывод информации о сбое
                    publishProgress(textFail);
                    break;

                case state.SEND:
                    //данные получены
                    publishProgress(waitDocs +": Ответ из ОИСМ получен "+ dbOism.length() +" байт");
                    mode = state.WRITE;
                    break;
            }

            //освобождение ресурсов
            postExecute();
            s = null;
            ip = null;
        }
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_ofd);

        //создание хендлера
        h = new OfdHandler();

        //получение отображателя текста
        tView = findViewById(R.id.ofd_exchange);
        tView.setMovementMethod(new ScrollingMovementMethod());
    }

    @Override
    protected void onStart() {
        super.onStart();

        //регистрация хендлера для получения ответов от контроллера ФН
        TMLib.getInstance().registerHandler(h);

        if ( !isStarted ) {
            isStarted = true;

            //останов фонового потока отправки в ОФД
            TMLib.getInstance().StopOFD();

            //останов фонового потока отправки в ОИСМ
            TMLib.getInstance().StopOISM();

            //запускаем процесс обращения к контроллеру ФН
            h.FRFNGetStatus();
        }
    }

    @Override
    protected void onStop() {
        //останов потоков обмена с сервером, если они запущены
        mode = state.ABORT;
        if ( sendOfdPack != null ) {
            sendOfdPack.cancel();
            sendOfdPack = null;
        }
        if ( sendOismPack != null ) {
            sendOismPack.cancel();
            sendOismPack = null;
        }

        if ( connStatus ) h.FRFNSetCommStatus(false); //сброс статуса соединения, если он установлен

        //освобождение хендлера
        TMLib.getInstance().unregisterHandler(h);

        super.onStop();
    }

    /** Обработка нажатия кнопки ОТМЕНА. */
    public void onButtonCancel(View view) {
        //завершение диалога
        isExit = true;
        finish();
    }

    /** Добавление сообщения в диалог активности. */
    private void addTextToView(String text) {
        tView.append(text+"\n");
    }

    /* Хендлер для обработки сообщений от TMLib. */
    private class OfdHandler extends TMLibHandler {

        /** Конструктор по умолчанию. */
        public OfdHandler(){
            super(MainApp.GetCtx().getMainLooper());
        }

        /** Получение настроек соединения с ОФД. */
        public void FROFDGetParam() {
            TMCommand cmd = new TMCommand();
            cmd.CmdOFDGetParam();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        public void FROFDGetParamExt() {
            TMCommand cmd = new TMCommand();
            cmd.CmdOFDGetParamExt(TMCommand.server_types.OISM);
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Получение информации об количестве неподтвержденных документов. */
        public void FRGetWaitDocs() {
            TMCommand cmd = new TMCommand();
            cmd.CmdGetWaitDocs();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Получение параметров регистрации ФН. */
        public void FRFNGetRegParam(short tlvId) {
            regTLV = tlvId;

            TMCommand cmd = new TMCommand();
            cmd.CmdFNGetRegParam(tlvId);
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Получение информации об ФН. */
        public void FRFNGetStatus() {
            TMCommand cmd = new TMCommand();
            cmd.CmdFNGetStatus();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Получение информации об состоянии обмена. */
        public void FRFNGetCommStatus() {
            TMCommand cmd = new TMCommand();
            cmd.CmdFNGetCommStatus();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Установка состоянии обмена. */
        public void FRFNSetCommStatus(boolean bConn) {
            TMCommand cmd = new TMCommand();
            cmd.CmdFNSetCommStatus(bConn);
            connStatus = bConn;
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Отмена чтения документа ФН для ОФД. */
        public void FRFNReadCancel() {
            TMCommand cmd = new TMCommand();
            cmd.CmdFNReadCancel();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Начало чтения документа ФН для ОФД. */
        public void FRFNReadOpen() {
            TMCommand cmd = new TMCommand();
            cmd.CmdFNReadOpen();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Чтение порции документа для ОФД. */
        public void FRFNRead() {
            TMCommand cmd = new TMCommand();
            cmd.CmdFNRead((short)(dbOfd.length()),(short)(Math.min(packetSize - dbOfd.length(), TMCommand.MAX_OFD_DATA_PART_SIZE)));
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Завершение чтения документа для ОФД. */
        public void FRFNReadClose() {
            TMCommand cmd = new TMCommand();
            cmd.CmdFNReadClose();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Сохранение полученной из ОФД квитанции в ФН. */
        public void FRFNWriteScript() {
            TMCommand cmd = new TMCommand();
            cmd.CmdFNWriteScript(dbOfd.getWithoutHeader());
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Получение количества документов для отправки в ОИСМ. */
        public void FRNotifGetStatus() {
            TMCommand cmd = new TMCommand();
            cmd.CmdNotifGetStatus();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Начало чтения документа ФН для ОИСМ. */
        public void FRNotifReadOpen() {
            TMCommand cmd = new TMCommand();
            cmd.CmdNotifReadOpen();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Отмена чтения документа ФН для ОИСМ. */
        public void FRNotifReadCancel() {
            TMCommand cmd = new TMCommand();
            cmd.CmdNotifReadCancel();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Чтение порции документа для ОФД. */
        public void FRNotifRead() {
            TMCommand cmd = new TMCommand();
            cmd.CmdNotifRead((short)(dbOism.length()),(short)(Math.min(packetSize - dbOism.length(), TMCommand.MAX_OFD_DATA_PART_SIZE)));
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Завершение чтения документа для ОФД. */
        public void FRNotifReadClose() {
            TMCommand cmd = new TMCommand();
            cmd.CmdNotifReadClose();
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        /** Сохранение полученной из ОИСМ квитанции в ФН. */
        public void FRNotifWriteReceipt() {
            TMCommand cmd = new TMCommand();
            cmd.CmdNotifWriteReceipt(dbOism.getWithoutHeader());
            TMLib.getInstance().DoCmd(OfdActivity.this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
        }

        @Override
        public void onRespCmd(TMCommand cmd) {
            //получен ответ на команду

            if ( isExit ) return; //если прервано пользователем, то обработка не нужна
            if ( (cmd.GetMainError() == 0) || (TMCommand.commands.CM_FNGetRegParam == (byte)cmd.GetCmdCode()) ){
                //обработка успешных ответов
                switch((byte) cmd.GetCmdCode()) {
                    case TMCommand.commands.CM_FNGetStatus:
                        //получаем заводской номер ФН
                        fnSN = cmd.GetFieldValue(11);
                        addTextToView("Заводской номер ФН: "+fnSN);
                        //определяем версию ФФД
                        FRFNGetRegParam(TMTag.NUM_VERSION_FFD);
                        break;

                    case TMCommand.commands.CM_FNGetRegParam:
                        switch(regTLV) {
                            case TMTag.NUM_VERSION_FFD:
                                //получена версия ФФД при регистрации ФН
                                ffdFN = cmd.GetFieldHexB(5);
                                addTextToView(String.format("ФН регистрирован по ФФД версии %d ", 0xFF & ffdFN) + ((ffdFN == 0) ? "(1.0)" : (ffdFN == TMCommand.proto_ffd.FFD_1_05) ? "(1.05)" : (ffdFN == TMCommand.proto_ffd.FFD_1_1) ? "(1.1)" : (ffdFN == TMCommand.proto_ffd.FFD_1_2) ? "(1.2)" : ""));
                                //чтение параметров ОФД
                                FROFDGetParam();
                                break;

                            case TMTag.KKT_REG_FLAGS:
                                //флаги регистрации
                                if ((cmd.GetMainError() == 0) && ((cmd.GetFieldHexDW(5) & TMCommand.kkt_mode_reg.TMT) != 0)) {
                                    //поддерживается маркировка
                                    addTextToView("ФН зарегистрирован с поддержкой ТМТ");
                                    //чтение параметров ОИСМ
                                    FROFDGetParamExt();
                                }
                                break;
                        }
                        break;

                    case TMCommand.commands.CM_OFDGetParam:
                        //получены параметры сервера
                        serverPort = 0xFFFF&cmd.GetFieldHexW(5);
                        serverConn = 0xFFFF&cmd.GetFieldHexW(6);
                        serverRecv = 0xFFFF&cmd.GetFieldHexW(7);
                        serverSend = 0xFFFF&cmd.GetFieldHexW(8);
                        serverAddress = cmd.GetFieldString(9);
                        if ( ( serverPort == 0 ) || ( serverConn == 0 ) || ( serverRecv == 0 ) || ( serverSend == 0 ) || ( serverAddress.length() == 0 ) ) {
                            addTextToView("ОШИБКА: Не настроены параметры сервера.");
                            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_bad_ofd_param), MessageBox.option.OK|MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
                        }
                        else {
                            addTextToView(((mode == state.PREPARE)?"ОФД":"ОИСМ")+" "+serverAddress+":"+ serverPort +" ("+ serverConn +","+ serverRecv +","+ serverSend +")");

                            //проверяем количество документов к отправке
                            if (mode == state.PREPARE)  FRGetWaitDocs();
                            else FRNotifGetStatus();
                        }
                        break;

                    case TMCommand.commands.CM_GetWaitDocs:
                        //получено количество документов на отправку
                        waitDocs = 0xFFFF&cmd.GetFieldHexW(5);
                        if ( waitDocs == 0 ) {
                            addTextToView("ВНИМАНИЕ: Нет сообщений для ОФД ожидающих отправку.");

                            if ( ffdFN >= TMCommand.proto_ffd.FFD_1_2 ) {
                                //меняем статус
                                mode = state.PREPARE_OISM;
                                //проверяем флаг регистрации TMT
                                FRFNGetRegParam(TMTag.KKT_REG_FLAGS);
                            }
                        }
                        else {
                            //запуск обмена с ОФД
                            FRFNGetCommStatus();
                        }
                        break;

                    case TMCommand.commands.CM_FNGetCommStatus:
                        //получен текущий статус обмена
                        if ( cmd.GetFieldHexB(6) != 0 ){
                            //нужно сбросить состояние документа
                            FRFNReadCancel();
                        }
                        else if ( (cmd.GetFieldHexB(5)&(TMCommand.ofd_status_flags.CONN_SET|TMCommand.ofd_status_flags.WAIT_RESP)) != 0 ) {
                            //нужно сбросить состояние обмена
                            FRFNSetCommStatus(false);
                        }
                        else {
                            waitDocs = 0xFFFF&cmd.GetFieldHexW(7);
                            if ( waitDocs == 0 ) {
                                addTextToView("ВНИМАНИЕ: Нет сообщений для ОФД ожидающих отправку.");
                            }
                            else {
                                mode = state.READ;
                                dbOfd = new TMOfd.DataBuf(fnSN,ffdFN);
                                //устанавливаем статус обмена
                                FRFNSetCommStatus(true);
                            }
                        }
                        break;

                    case TMCommand.commands.CM_FNReadCancel:
                        switch ( mode ){
                            case state.PREPARE:
                                //возвращаемся к получению статуса
                                FRFNGetCommStatus();
                                break;

                            case state.READ:
                                //ошибка при чтении сообщения
                                mode = state.FAIL;
                                //сбрасываем состояние обмена
                                FRFNSetCommStatus(false);
                                break;
                        }
                        break;

                    case TMCommand.commands.CM_FNReadOpen:
                        packetSize = 0xFFFF&cmd.GetFieldHexW(5);
                        if ( packetSize == 0 ) {
                            //ошибка: пакет не может быть нулевого размера
                            addTextToView("ОШИБКА: Размер пакета не может быть нулевым.");
                            FRFNReadCancel();
                        }
                        else {
                            //чтение пакета
                            addTextToView(waitDocs +": Чтение пакета из ФН "+ packetSize +" байт.");
                            FRFNRead();
                        }
                        break;

                    case TMCommand.commands.CM_FNRead: {
                        byte[] data = cmd.GetFieldBin(5);
                        if (data != null) dbOfd.add(data);
                        if (dbOfd.length() < packetSize) {
                            //продолжаем читать
                            FRFNRead();
                        } else {
                            //отправка пакета
                            FRFNReadClose();
                        }
                        break;
                    }

                    case TMCommand.commands.CM_FNReadClose:
                        //отправка пакета
                        mode = state.SEND;
                        addTextToView(waitDocs +": Отправка пакета в ОФД "+ packetSize +" байт.");
                        sendOfdPack = new SendOfdPacket();
                        sendOfdPack.start();
                        break;

                    case TMCommand.commands.CM_FNWriteScript: {
                        //зацикливаем опять переходим к чтению состояния
                        int code = cmd.GetFieldHexW(5);
                        String msg = cmd.GetFieldString(6);
                        mode = state.PREPARE;
                        addTextToView(waitDocs + ": Квитанция сохранена в ФН (код " + code + ").");
                        if ((msg != null) && (msg.length() > 0))
                            addTextToView(waitDocs + ": Сообщение '" + msg + "'.");
                        FRFNGetCommStatus();
                        break;
                    }

                    case TMCommand.commands.CM_FNSetCommStatus:
                        switch ( mode ){
                            case state.PREPARE:
                                //повторяем  получение статуса
                                FRFNGetCommStatus();
                                break;

                            case state.READ:
                                //начинаем процесс чтения документа
                                FRFNReadOpen();
                                break;
                        }
                        break;

                    case TMCommand.commands.CM_NotifGetStatus:
                        if ( cmd.GetFieldHexB(5) != 0 ) {
                            addTextToView("ОШИБКА: Неверное состояние чтения уведомления.");
                            FRNotifReadCancel();
                        }
                        else {
                            waitDocs = cmd.GetFieldHexW(6);
                            if (waitDocs == 0) {
                                addTextToView("ВНИМАНИЕ: Нет сообщений для ОИСМ ожидающих отправку.");
                            } else {
                                //начинаем читать документ
                                mode = state.READ;
                                dbOism = new TMOism.DataBuf(TMOism.DataBuf.OISM_THDR_PROTOCOL, fnSN);
                                FRNotifReadOpen();
                            }
                        }
                        break;

                    case TMCommand.commands.CM_NotifReadOpen:
                        packetSize = 0xFFFF&cmd.GetFieldHexW(5);
                        if ( packetSize == 0 ) {
                            //ошибка: пакет не может быть нулевого размера
                            addTextToView("ОШИБКА: Размер пакета не может быть нулевым.");
                            FRNotifReadCancel();
                        }
                        else {
                            //чтение пакета
                            addTextToView(waitDocs +": Чтение пакета из ФН "+ packetSize +" байт.");
                            FRNotifRead();
                        }
                        break;

                    case TMCommand.commands.CM_NotifRead: {
                        byte[] data = cmd.GetFieldBin(5);
                        if (data != null) dbOism.add(data);
                        if (dbOism.length() < packetSize) {
                            //продолжаем читать
                            FRNotifRead();
                        } else {
                            //отправка пакета
                            FRNotifReadClose();
                        }
                        break;
                    }

                    case TMCommand.commands.CM_NotifReadClose:
                        //отправка пакета
                        mode = state.SEND;
                        addTextToView(waitDocs +": Отправка пакета в ОИСМ "+ packetSize +" байт.");
                        sendOismPack = new SendOismPacket();
                        sendOismPack.start();
                        break;

                    case TMCommand.commands.CM_NotifWriteReceipt: {
                        //зацикливаем опять переходим к чтению состояния
                        String msg = cmd.GetFieldString(5);
                        mode = state.PREPARE_OISM;
                        addTextToView(waitDocs + ": Квитанция сохранена в ФН (тег 2106:" + String.format("%02X",0xFF&cmd.GetFieldHexW(6)) + ", тег 2111:"+ String.format("%02X",0xFF&cmd.GetFieldHexW(7)) + ").");
                        if ((msg != null) && (!msg.isEmpty()))
                            addTextToView(waitDocs + ": Сообщение '" + msg + "'.");
                        FRNotifGetStatus();
                        break;
                    }
                }
            }
            else {
                //обработка ошибок
                if ( cmd.GetCmdCode() != TMCommand.commands.CM_FNSetCommStatus ) {
                    addTextToView(String.format(getString(R.string.msg_cmd_error), cmd.GetMainError(), cmd.GetSubError(), cmd.GetCmdCode()) + ":" + TMError.GetText((byte) cmd.GetMainError()));
                    if ( mode != state.PREPARE ) {
                        //перезапускаем
                        mode = state.PREPARE;
                        FRFNGetCommStatus();
                    }
                }
                else {
                    FRGetWaitDocs();
                }
            }
        }

        @Override
        public void onTimeOut() {
            //сообщение о превышении таймаута обращения к контроллеру ФН
            if ( mode == state.ABORT ) return; //если прервано пользователем, то обработка не нужна
            addTextToView("ОШИБКА: Превышен таймаут обращения к сервису контроллера ФН.");
        }

        @Override
        public void onError(String errText) {
            //сообщение о сбое сервиса взаимодействия с контроллером ФН
            if ( mode == state.ABORT ) return; //если прервано пользователем, то обработка не нужна
            addTextToView("ОШИБКА: Сбой сервиса взаимодействия с контроллером ФН - " + errText);
        }
    }
}
