package ru.nilsoft.example;

import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.os.Handler;
import androidx.appcompat.app.AppCompatActivity;
import android.os.Bundle;
import android.view.View;
import android.widget.TextView;

import java.io.UnsupportedEncodingException;

import ru.nilsoft.tm.TMCommand;
import ru.nilsoft.tm.TMError;
import ru.nilsoft.tm.TMLib;
import ru.nilsoft.tm.TMLibHandler;

/**
 * Печать нефискальный документа.
 *
 * Демонстрируются два вида печати нефискального документа:
 * - Прямая печать через библиотеку работы с сервисом ФН;
 * - С использованием печати через команды контроллера ФН.
 *
 * @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class NonFiscalActivity extends AppCompatActivity implements MessageBox.CallBack{
    /** Handler для работы c контроллером ФН. */
    private Handler h;
    /** Режим прямой печати. */
    private boolean isDirect = false;

    /** Названия настроек для сохранения в файл настроек приложения */
    public static final class sets{
        /** Имя пользователя. */
        static final String TEXT = "NonFiscal_Text";
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_non_fiscal);

        //создание хендлера
        h = new LibHandler();
    }

    @Override
    protected void onStart() {
        super.onStart();

        //регистрация хендлера для получения ответов от контроллера ФН
        TMLib.getInstance().registerHandler(h);

        //заполнение данными (считываются из файла настроек приложения)
        ((TextView)findViewById(R.id.text_to_print)).setText(MainApp.settings.getString(sets.TEXT,
                "Тестовый текст для проверки печати.\nЧек включает до 40 символов в строке.\nВысота и ширина знаков могут удваиваться.\nСимволы могут инвертироваться."));
    }

    @Override
    protected void onStop() {
        //освобождение интерфейса
        TMLib.getInstance().unregisterHandler(h);

        //сохранение данных (сохранение в файл настроек приложения)
        SharedPreferences.Editor editor = MainApp.settings.edit();
        editor.putString(sets.TEXT,String.valueOf(((TextView)findViewById(R.id.text_to_print)).getText()));
        editor.apply();

        super.onStop();
    }

    public void onMessageBoxPositiveClick(DialogInterface dialog) {
    }

    public void onMessageBoxNegativeClick(DialogInterface dialog) {
    }

    /** Обработка нажатия кнопки ПЕЧАТЬ КАК РЕПОРТ. */
    public void onButtonReport(View view) {
        FRNonFiscalOpen();
    }

    /** Обработка нажатия кнопки ПРОИЗВОЛЬНАЯ ПЕЧАТЬ. */
    public void onButtonPrintDirect(View view) {
        String data = String.valueOf(((TextView)findViewById(R.id.text_to_print)).getText());
        byte[] tmp;
        try {
            //данные для печати преобразовываются в CP866
            tmp = data.getBytes("Cp866");
        } catch (UnsupportedEncodingException e) {
            tmp = data.getBytes();
        }
        isDirect = true;
        //выполнение команды печати
        TMLib.getInstance().SendPrintData(tmp);
    }

    /** Обработка нажатия кнопки ОТМЕНА. */
    public void onButtonCancel(View view) {
        finish(); //завершение диалога
    }

    /** Команда открытия нефискального отчета. */
    private void FRNonFiscalOpen() {
        TMCommand cmd = new TMCommand();
        cmd.CmdNonFiscalOpen();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Команда добавление строк отчета.
     * @param t строки.
     */
    private void FRNonFiscalLines(String[] t) {
        TMCommand cmd = new TMCommand();
        cmd.CmdNonFiscalLines(t);
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /** Команда закрытия нефискального отчета. */
    private void FRNonFiscalClose() {
        TMCommand cmd = new TMCommand();
        cmd.CmdNonFiscalClose();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_PRINT));
    }

    /** Отмена документа. */
    private void FRReceiptCancel() {
        TMCommand cmd = new TMCommand();
        cmd.CmdReceiptCancel();
        TMLib.getInstance().DoCmd(null,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /* Хендлер для обработки сообщений от контроллера ФН. */
    private class LibHandler extends TMLibHandler {

        /** Конструктор по умолчанию. */
        public LibHandler(){
            super(MainApp.GetCtx().getMainLooper());
        }

        @Override
        public void onRespCmd(TMCommand cmd) {
            //получен ответ на команду
            if (cmd.GetMainError() == 0) {
                //обработка успешных ответов
                switch ((byte) cmd.GetCmdCode()) {
                    case TMCommand.commands.CM_NonFiscalOpen:
                        //документ успешно открыт
                        String[] data = String.valueOf(((TextView) findViewById(R.id.text_to_print)).getText()).split("\n");
                        //добавление строк
                        FRNonFiscalLines(data);
                        break;

                    case TMCommand.commands.CM_NonFiscalLines:
                        //закрытие документа
                        FRNonFiscalClose();
                        break;

                    case TMCommand.commands.CM_NonFiscalClose:
                        //документ напечатан
                        MessageBox.Create(getString(R.string.header_success), getString(R.string.msg_cmd_success)+"\n\nЗАБЕРИТЕ ЧЕК", MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
                        break;
                }
            } else {
                //ошибка при выполнении команды
                if ( cmd.GetCmdCode() != TMCommand.commands.CM_ReceiptCancel ) {
                    //отмена чека
                    FRReceiptCancel();
                    //выводим сообщение об ошибке
                    MessageBox.Create(getString(R.string.header_error), String.format(getString(R.string.msg_cmd_error), cmd.GetMainError(), cmd.GetSubError(), cmd.GetCmdCode()) + "\n\n" + TMError.GetText((byte) cmd.GetMainError()), MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
                }
            }
        }

        @Override
        public void onTimeOut() {
            //сообщение о превышении таймаута обращения к контроллеру ФН
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_timeout), MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }

        @Override
        public void onError(String errText) {
            //сообщение о сбое сервиса взаимодействия с контроллером ФН
            isDirect = false;
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_service_fail)+ "\n\n" + errText, MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }

        @Override
        public void onPrint(int mainErr, int subErr) {
            //получен ответ от модуля печати

            if ((mainErr == 0) && (isDirect)) {
                //отрезать бумагу
                TMLib.getInstance().SendPrintCut();
                MessageBox.Create(getString(R.string.header_success), getString(R.string.msg_print_success), MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
            }

            isDirect = false;

            if (mainErr != 0) {
                //сообщение об ошибке печати
                MessageBox.Create(getString(R.string.header_error), String.format(getString(R.string.msg_print_error), mainErr, subErr)+"\n\n"+TMError.GetPrnText(mainErr, subErr), MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
            }
        }
    }
}
