package ru.nilsoft.example;

import android.app.AlertDialog;
import android.app.Dialog;
import android.content.DialogInterface;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.os.Bundle;
import android.view.KeyEvent;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;

/**
 * Диалог для быстрого вывода сообщений.
 *
 *  @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class MessageBox extends DialogFragment {
    /** Выводимое сообщение. */
    private String message;
    /** Заголовок сообщения. */
    private String header;
    /** Опции. */
    private int options;
    /** Callback listener. */
    private CallBack listener;

    /** Таг для вывода. */
    public static final String TAG = "MessageBox";

    /** Опции диалога. */
    public static class option{
        /** Есть кнопка ОК. */
        static final int OK     = 0x0001;
        /** Есть кнопка ОТМЕНА. */
        static final int CANCEL = 0x0002;
        /** Диалог предупреждение. */
        static final int WARN   = 0x0004;
        /** Диалог сообщение об ошибкке. */
        static final int ERROR  = 0x0008;
        /** Диалог Да Нет. */
        static final int YES_NO = 0x0010;
    }

    /** Определение интерфейса обратного вызова. */
    public interface CallBack {
        /** Нажатие ОК. */
        void onMessageBoxPositiveClick(DialogInterface dialog);
        /** Нажатие ОТМЕНА. */
        void onMessageBoxNegativeClick(DialogInterface dialog);
    }

    /** Получение и создание диалога. */
    static public MessageBox Create(String hd, String msg, int opt){
        MessageBox m = new MessageBox();
        m.message = msg;
        m.header = hd;
        m.options = opt;
        m.listener = null;
        return m;
    }

    /** Получение и создание диалога (с listener'ом). */
    static public MessageBox Create(String hd, String msg, int opt, CallBack l){
        MessageBox m = new MessageBox();
        m.message = msg;
        m.header = hd;
        m.options = opt;
        m.listener = l;
        return m;
    }

    @Override
    public Dialog onCreateDialog(Bundle savedInstanceState) {
        // Use the Builder class for convenient dialog construction
        AlertDialog.Builder builder = new AlertDialog.Builder(requireActivity());
        //установка заголовка
        if ( header != null ) builder.setTitle(header);
        //установка сообщения
        builder.setMessage(message);
        //защита от кнопки "назад"
        builder.setOnKeyListener((dialog, keyCode, event) -> {
            return (keyCode == KeyEvent.KEYCODE_BACK) && (event.getAction() == KeyEvent.ACTION_UP);
        });
        //установка кнопки ОК
        if ( ( (options&option.OK) != 0 ) || ( (options&option.CANCEL) == 0 ) ) {
            builder.setPositiveButton((((options&option.YES_NO)==0)?getString(R.string.btn_ok):getString(R.string.btn_yes)), new DialogInterface.OnClickListener() {
                public void onClick(DialogInterface dialog, int id) {
                    // убеждаемся, что активность содержит обратный интерфейс
                    if ( ( listener == null ) && ( requireActivity() instanceof CallBack ) ) listener = (CallBack)requireActivity();
                    if ( listener != null ) listener.onMessageBoxPositiveClick(dialog);
                }
            });
        }
        //установка кнопки ОТМЕНА
        if ( (options&option.CANCEL) != 0 ) {
            builder.setNegativeButton((((options&option.YES_NO)==0)?getString(R.string.btn_cancel):getString(R.string.btn_no)), new DialogInterface.OnClickListener() {
                public void onClick(DialogInterface dialog, int id) {
                    // убеждаемся, что активность содержит обратный интерфейс
                    if ( ( listener == null ) && ( requireActivity() instanceof CallBack ) ) listener = (CallBack)requireActivity();
                    if ( listener != null ) listener.onMessageBoxNegativeClick(dialog);
                }
            });
        }
        // Create the AlertDialog object and return it
        Dialog dlg = builder.create();
        if ( (options&option.WARN) != 0 ) dlg.getWindow().setBackgroundDrawable(new ColorDrawable(Color.YELLOW));
        if ( (options&option.ERROR) != 0 ) dlg.getWindow().setBackgroundDrawable(new ColorDrawable(Color.RED));
        dlg.setCanceledOnTouchOutside(false);
        dlg.setCancelable(false);
        return dlg;
    }

    @Override
    public void show(@NonNull FragmentManager manager, @Nullable String tag) {
        FragmentTransaction ft = manager.beginTransaction();
        ft.add(this, tag);
        ft.commitAllowingStateLoss();
    }
}
