package ru.nilsoft.example;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.Toolbar;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentPagerAdapter;
import androidx.lifecycle.Lifecycle;
import androidx.viewpager.widget.ViewPager;
import androidx.viewpager2.adapter.FragmentStateAdapter;
import androidx.viewpager2.widget.ViewPager2;

import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;

import com.google.android.material.tabs.TabLayout;
import com.google.android.material.tabs.TabLayoutMediator;

import org.jetbrains.annotations.Nullable;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import ru.nilsoft.tm.TMCommand;
import ru.nilsoft.tm.TMError;
import ru.nilsoft.tm.TMLib;
import ru.nilsoft.tm.TMLibHandler;

/**
 * Главная активность приложения.
 *
 * При старте приложения производит инициализацию контроллера ФН.
 * Процесс инициализации:
 * - Отправка команды DLE 9, для определения в каком режиме находится контроллер;
 * - Отправка команды DLE 0, для проверки что контроллер уже активирован;
 * - Активация контроллера (DLE 7), если требуется активировать контроллер;
 * - Отправка пустой команды (0x70) для получения статуса контроллера;
 * - Отправка команды старт сессии (0x01), если сессия не открыта;
 * - Проверка расхождения времени терминала и контроллера ФН;
 *
 * @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class MainActivity extends AppCompatActivity implements MessageBox.CallBack {
    /** Handler для работы c libTM. */
    private Handler h;
    /** Номер выполняемой команды, используется для составных и DLE команд. */
    private int cmdNum = 0;
    /** Если попытка активации уже была проведена. */
    private boolean isActivate = false;
    /** Сумма (для операций с виртуальным денежным ящиком). */
    private String operSum = "";
    /** ФИО кассира (для операций с виртуальным денежным ящиком). */
    private String operFIO = "";
    /** UTC контроллера ФН, полученный от сервиса ФН. */
    private int fnUTC = 3; //поумолчанию часовой пояс +3 (Москва)

    /** Контракт для получения результата проверки пермишенов. */
    private final ActivityResultLauncher<Intent> resultPermissions = registerForActivityResult(new ActivityResultContracts.StartActivityForResult(), result -> {
        // обработка result - пофиг на результат
        if ( PermissionsActivity.check(this) ) StartController();
    });

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        try {
            //создание механизма работы с вкладками
            Toolbar toolbar = findViewById(R.id.toolbar);
            setSupportActionBar(toolbar);

            //привязка пролистовщика с адаптером
            ViewPager2 mViewPager = findViewById(R.id.container);
            //mViewPager.setSaveEnabled(false); //чтобы не падало при подключении клавиатуры или сканера
            //адаптер вложенных фрагментов для тулбара
            mViewPager.setAdapter(new ViewStateAdapter(getSupportFragmentManager(), getLifecycle()));

            //связка таба с пролистовщиком
            TabLayout tabLayout = findViewById(R.id.tabs);
            //ВАЖНО: smoothScroll надо отключать - глючит на Тактилионе (ставит всегда на последнюю позицию ViewPager2)
            new TabLayoutMediator(tabLayout, mViewPager, true,false,
                    (tab, position) -> tab.setText(getResources().getStringArray(R.array.tab_items)[position])
            ).attach();
        }
        catch(Exception ignored){}

        //инициализация интерфейса для работы с КФН
        //Log.d("example","MainActivity init TMService: "+TMLib.getInstance().init(getApplicationContext()));

        //создание хендлера
        h = new LibHandler();

        //получение сообщений от диалогов работы с денежным ящиком
        getSupportFragmentManager().setFragmentResultListener(CashBoxDialog.RESULT_KEY, this, (requestKey, result) -> {
            try {
                //нода оригинального чека
                boolean mode = result.getBoolean("mode");
                operSum = result.getString("operSum");
                operFIO = result.getString("operFIO");

                //регистрация хендлера
                TMLib.getInstance().registerHandler(h);

                //составная операция, сперва получаем время из КФН
                cmdNum = mode?TMCommand.commands.CM_FromCash:TMCommand.commands.CM_ToCash;
                FRGetDateTime();
            }
            catch (Exception ignored) {
            }
        });
    }

    @Override
    protected void onStart() {
        super.onStart();

//        if (!PermissionsActivity.check(this)) {
//            //запрос прав
//            Intent intent = new Intent(this, PermissionsActivity.class);
//            resultPermissions.launch(intent);
//        } else {
            //запускаем процесс инициализации
            StartController();
//        }
    }

    @Override
    protected void onStop() {
        //освобождение интерфейса
        TMLib.getInstance().unregisterHandler(h);

        super.onStop();
    }

    /**
     * Запуск иницализации контроллера ФН.
     */
    private void StartController(){
        //регистрация хендлера
        TMLib.getInstance().registerHandler(h);

        //если контроллер не инициализирован, то запускаем процесс инициализации
        if ( ( TMLib.getInstance().getState(TMLib.states.READY) != 0 ) && ( TMLib.getInstance().getState(TMLib.states.INIT) == 0 ) ) FR_DLE9();

        //чтобы получить текущий часовой пояс настроенный для контроллера ФН
        TMLib.getInstance().SendRespVer();
    }

    public void onMessageBoxPositiveClick(DialogInterface dialog) {
        if ( cmdNum == TMCommand.commands.CM_GetDateTime ) {
            //установка времени в КФН
            onButtonSetTime(null);
        }
    }

    public void onMessageBoxNegativeClick(DialogInterface dialog) {
    }

    /** Класс для создания фрагментов. */
    public static class MainFragment extends Fragment {
        protected int resId;

        public MainFragment() {
        }

        public static MainFragment newInstance(int ResId) {
            MainFragment fragment = new MainFragment();
            fragment.resId = ResId;
            return fragment;
        }

        @Override
        public View onCreateView(@NonNull LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
            // Inflate the layout for this fragment
            if( resId != 0 )return inflater.inflate(resId, container, false);
            return null;
        }

        /**
         * Для восстановления состояния при повороте.
         * @param savedInstanceState пакет.
         */
        @Override
        public void onCreate(@Nullable Bundle savedInstanceState) {
            super.onCreate(savedInstanceState);

            if ( (savedInstanceState != null) && savedInstanceState.containsKey("resId") ) resId = savedInstanceState.getInt("resId");
        }

        /**
         * Для сохранения текста при повороте.
         * @param outState пакет.
         */
        @Override
        public void onSaveInstanceState(@NonNull Bundle outState) {
            //сохранение данных
            outState.putInt("resId",resId);

            super.onSaveInstanceState(outState);
        }
    }

    /** Адаптер страниц-фрагментов главного меню. */
    public static class ViewStateAdapter extends FragmentStateAdapter {
        private static final MainFragment [] fragments = {
            MainFragment.newInstance(R.layout.fragment_main_shift),
            MainFragment.newInstance(R.layout.fragment_main_receipt),
            MainFragment.newInstance(R.layout.fragment_main_ofd),
            MainFragment.newInstance(R.layout.fragment_main_sets),
            MainFragment.newInstance(R.layout.fragment_main_info),
        };

        Fragment lastFragment = null;

        /**
         * Конструктор адаптера для главного меню.
         * @param fragmentManager манаджер фрагментов.
         * @param lifecycle жизненный путь приложения.
         */
        public ViewStateAdapter(@NonNull FragmentManager fragmentManager, @NonNull Lifecycle lifecycle) {
            super(fragmentManager, lifecycle);
        }

        /**
         * Создание фрагмента по позиции.
         * @param position позиция.
         * @return фрагмент.
         */
        @NonNull
        @Override
        public Fragment createFragment(int position) {
            //Log.d(this.getClass().getSimpleName(),"position "+position);
            if ( (position >= 0) && (position < getItemCount()) ) return lastFragment = fragments[position];
            return lastFragment = fragments[0];
        }

        /**
         * Количество позиций в меню.
         * @return количество позиций.
         */
        @Override
        public int getItemCount() {
            //количество определенных вкладок
            return fragments.length;
        }
    }

    /** Обработка нажатия кнопки ИНФОРМАЦИЯ. */
    public void onButtonInfo(View view) {
        Intent intent = new Intent(this, InfoActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ПРОМЕЖУТОЧНЫЙ ОТЧЕТ. */
    public void onButtonXReport(View view) {
        //составная команда, сперва считываем время КФН
        cmdNum = TMCommand.commands.CM_ShiftReport;
        FRGetDateTime();
    }

    /** Обработка нажатия кнопки ОТКРЫТЬ СМЕНУ. */
    public void onButtonOpenShift(View view) {
        Intent intent = new Intent(this, OpenShiftActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ЗАКРЫТЬ СМЕНУ. */
    public void onButtonCloseShift(View view) {
        Intent intent = new Intent(this, CloseShiftActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ПАРАМЕТРЫ СЕРВЕРОВ. */
    public void onButtonSetOfd(View view) {
        Intent intent = new Intent(this, SetOfdActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ОТПРАВКА В ОФД. */
    public void onButtonOfd(View view) {
        Intent intent = new Intent(this, OfdActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ПАРАМЕТРЫ ДОКУМЕНТОВ. */
    public void onButtonDocumentsParams(View view) {
        Intent intent = new Intent(this, DocumentsParamsActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки СЧЕТЧИКИ ДОКУМЕНТОВ. */
    public void onButtonCounters(View view) {
        Intent intent = new Intent(this, CountersActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ЭЛЕКТРОННЫЙ ОТЧЕТ. */
    public void onButtonMoney(View view) {
        Intent intent = new Intent(this, MoneyActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ЧЕК. */
    public void onButtonReceipt(View view) {
        Intent intent = new Intent(this, ReceiptActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ПРОИЗВОЛЬНЫЙ ЧЕК. */
    public void onButtonReceiptFree(View view) {
        Intent intent = new Intent(this, ReceiptFreeActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки НЕФИСКАЛЬНЫЙ ЧЕК. */
    public void onButtonNonFiscal(View view) {
        Intent intent = new Intent(this, NonFiscalActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ЗАГОЛОВОК/ОКОНЧАНИЕ ЧЕКА. */
    public void onButtonHeader(View view) {
        Intent intent = new Intent(this, HeaderActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки СИНХРОНИЗАЦИЯ ВРЕМЕНИ. */
    public void onButtonSetTime(View view) {
        Intent intent = new Intent(this, SetTimeActivity.class);
        startActivity(intent);
    }

    /** Обработка нажатия кнопки ОТМЕНА ЧЕКА. */
    public void onButtonReceiptCancel(View view) { FRReceiptCancel(); }


    /** Обработка нажатия кнопки ПОДКРЕПЛЕНИЕ. */
    public void onButtonToCash(View view) {
        CashBoxDialog.Create(false).show(getSupportFragmentManager(), CashBoxDialog.TAG);
    }

    /** Обработка нажатия кнопки ИНКАССАЦИЯ. */
    public void onButtonFromCash(View view) {
        CashBoxDialog.Create(true).show(getSupportFragmentManager(), CashBoxDialog.TAG);
 
    }

    /** Проверка режима контроллера ФН. */
    private void FR_DLE9() {
        if ( TMLib.getInstance().getState(TMLib.states.READY) != 0 ) {
            cmdNum = 0x1039;
            TMLib.getInstance().DoDLE(this,(byte)0x39,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_STATUS));
        }
    }

    /** Проверка состояния контроллера ФН. */
    private void FR_DLE0() {
        if ( TMLib.getInstance().getState(TMLib.states.READY) != 0 ) {
            cmdNum = 0x1030;
            //при инициализации таймаут может до 10 секунд, поэтому ставим таймаут с запасом
            TMLib.getInstance().DoDLE(this,(byte)0x30,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_PRINT));
        }
    }

    /** Активация контроллера ФН. */
    private void FRActivation() {
        if ( TMLib.getInstance().getState(TMLib.states.INIT) == 0 ) {
            TMLib.getInstance().Activate(this,6*CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
            isActivate = true;
        }
    }

    /** Команда начала сессии. */
    private void FRStart(String tDate, String tTime) {
        TMCommand cmd = new TMCommand();
        if ( ( tDate != null ) && ( tTime != null ) ) cmd.CmdStart(tDate,tTime);
        else cmd.CmdStart();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /** Команда получения времени. */
    private void FRGetDateTime() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetDateTime();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /** Команда отчета смены. */
    private void FRShiftReport(String tDate, String tTime) {
        TMCommand cmd = new TMCommand();
        if ( ( tDate != null ) && ( tTime != null ) ) cmd.CmdShiftReport(tDate,tTime);
        else cmd.CmdShiftReport();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_PRINT));
    }

    /** Команда отмены чека. */
    private void FRReceiptCancel() {
        TMCommand cmd = new TMCommand();
        cmd.CmdReceiptCancel();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /** Команда внесения средств. */
    private void FRToCash(String tDate, String tTime, String sum, String fio) {
        TMCommand cmd = new TMCommand();
        if ( ( tDate != null ) && ( tTime != null ) ) cmd.CmdToCash(tDate,tTime,sum,fio);
        else cmd.CmdToCash(sum,fio);
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_PRINT));
    }

    /** Команда изъятия средств. */
    private void FRFromCash(String tDate, String tTime, String sum, String fio) {
        TMCommand cmd = new TMCommand();
        if ( ( tDate != null ) && ( tTime != null ) ) cmd.CmdFromCash(tDate,tTime,sum,fio);
        else cmd.CmdFromCash(sum,fio);
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_PRINT));
    }

    /** Получение информации об количестве неподтвержденных документов. */
    public void FRGetWaitDocs() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetWaitDocs();
        TMLib.getInstance().DoCmd(this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
    }

    /** Получение информации о КФН. */
    public void FRGetInfo() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetInfo();
        TMLib.getInstance().DoCmd(this, cmd, 2*CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
    }

    /** Хендлер для обработки сообщений от TMLib. */
    private class LibHandler extends TMLibHandler{

        /** Конструктор по умолчанию. */
        public LibHandler(){
            super(MainApp.GetCtx().getMainLooper());
        }

        @Override
        public void onVersion(Bundle data, int hi_ver, int mi_ver, int lo_ver, int date_ver, String text) {
            //получена версия сервиса ККТ
            fnUTC = data.getInt("utc",3); //по умолчанию делаем по МОСКВЕ
        }

        @Override
        public void onReady() {
            //соединение с сервисом установлено
            FR_DLE9();
        }

        @Override
        public void onActive() {
            MainApp.MakeNotification("инициализация прошла успешно");
            //активация прошла успешно, проверяем состояние контроллера
            FR_DLE9();
        }

        @Override
        public void onTimeOut() {
            int lastCmdNum = cmdNum;
            cmdNum = 0;
            switch( lastCmdNum ) {
                case 0x1039:
                case 0x1030:
                    //при сбое DLE команд, необходима инициализация
                    if ( !isActivate ) {
                        FRActivation();
                        break;
                    }

                default:
                    //сообщение о превышении таймаута
                    MessageBox.Create(getString(R.string.header_error),getString(R.string.msg_timeout)+"\n"+TMLib.getInstance().getLastCmd(), MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
            }
        }

        @Override
        public void onError(String errText) {
            cmdNum = 0;
            //сообщение о сбое сервиса
            MessageBox.Create(getString(R.string.header_error), getString(R.string.msg_service_fail)+"\n\n"+errText, MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
        }

        @Override
        public void onRespData(byte [] data) {
            //ответ на DLE команду
            int lastCmdNum = cmdNum;
            cmdNum = 0;
            switch( lastCmdNum ) {
                case 0x1039:
                    //проверяем последний полученный байт
                    if ( ( data.length > 0 ) && ( data[data.length-1] != 0x12 ) && ( (data[data.length-1]&0x04) != 0 ) )
                    {
                        //по умолчанию x12 (по маске x93) - 2ой бит: 1 - режим монитор, 3ий бит: 1 - режим монитора по команде; 5ий бит:1 - перемычка; 6ый бит:1 - ошибка
                        //выходим с ошибкой контроллер в сервисном режиме
                        cmdNum = lastCmdNum; //чтобы обработалось сообщение
                        MessageBox.Create(getString(R.string.header_warning), String.format(getString(R.string.msg_in_service_mode), data[0]), MessageBox.option.OK | MessageBox.option.CANCEL | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
                    }
                    else {
                        //проверка состояния контроллера ФН
                        FR_DLE0();
                    }
                    break;

                case 0x1030:
                    if ( TMLib.getInstance().getState(TMLib.states.INIT) == 0 ) {
                        //необходима инициализация
                        FRActivation();
                    }
                    else {
                        //для проверки статусов контроллера
                        FRGetInfo();
                    }
                    break;
            }
        }

        @Override
        public void onRespCmd(TMCommand cmd) {
            int lastCmdNum = cmdNum;
            cmdNum = 0;

            if ( (cmd.GetMainError() != 0) && ((byte)cmd.GetCmdCode() != TMCommand.commands.CM_Start) ) {
                //сообщение об ошибке выполнения команды
                MessageBox.Create(getString(R.string.header_error), String.format(getString(R.string.msg_cmd_error), cmd.GetMainError(), cmd.GetSubError(), cmd.GetCmdCode()) + "\n\n" + TMError.GetText((byte) cmd.GetMainError()), MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
            }
            else {
                //команда успешно выполнена
                switch ((byte) cmd.GetCmdCode()) {
                    case TMCommand.commands.CM_GetInfo:
                        //проверяем статусы чтобы узнать нужно ли подавать команду старт
                        if ( (cmd.GetDynamicStatus()&TMCommand.dynamic_status_flags.SESSION_NOT_CLOSE) == 0 ) {
                            //команда старт требует текущее время, поэтому запрашиваем время из контроллера
                            cmdNum = TMCommand.commands.CM_Start;
                        }

                        try {
                            //вывод серийного номера в заголовок окна
                            getSupportActionBar().setTitle(getString(R.string.app_name) + " " + getString(R.string.text_view_kkt_sn) + cmd.GetFieldValue(7));
                        }
                        catch(Exception ignored) {
                        }

                        FRGetDateTime();
                        break;

                    case TMCommand.commands.CM_Start:
                        //команда успешно выполнена, спрашиваем текущее время КФН
                        FRGetDateTime();
                        break;

                    case TMCommand.commands.CM_GetDateTime:
                        switch((byte) lastCmdNum) {
                            case TMCommand.commands.CM_ShiftReport:
                                //время получено, выполняем команду
                                FRShiftReport(cmd.GetFieldValue(5),cmd.GetFieldValue(6));
                                break;

                            case TMCommand.commands.CM_Start:
                                //время получено, выполняем команду
                                FRStart(cmd.GetFieldValue(5),cmd.GetFieldValue(6));
                                break;

                            case TMCommand.commands.CM_ToCash:
                                //время получено, выполняем команду
                                FRToCash(cmd.GetFieldValue(5),cmd.GetFieldValue(6),operSum,operFIO);
                                break;

                            case TMCommand.commands.CM_FromCash:
                                //время получено, выполняем команду
                                FRFromCash(cmd.GetFieldValue(5),cmd.GetFieldValue(6),operSum,operFIO);
                                break;

                            default: {
                                //сравнивание времени в андроид и контроллере
                                Calendar cal = Calendar.getInstance();
                                try {
                                    String FRdt = cmd.GetFieldValue(5) + " " + cmd.GetFieldValue(6)+ " " + String.format("GMT%s%02d:00",((fnUTC<0)?"-":"+"),fnUTC);
                                    String Cdt = String.format("%02d%02d%02d %02d%02d GMT%s%02d:%02d",cal.get(Calendar.DAY_OF_MONTH),cal.get(Calendar.MONTH)+1,cal.get(Calendar.YEAR)%100,cal.get(Calendar.HOUR_OF_DAY),cal.get(Calendar.MINUTE),((cal.get(Calendar.ZONE_OFFSET)<0)?"-":"+"),cal.get(Calendar.ZONE_OFFSET)/(60*60*1000),(cal.get(Calendar.ZONE_OFFSET)/(60*1000))%60);
                                    SimpleDateFormat df = new SimpleDateFormat("ddMMyy HHmm z");
                                    Date dt;
                                    dt = df.parse(FRdt);
                                    long frTime = dt.getTime();
                                    long curTime = cal.getTimeInMillis();
                                    long diff = (frTime > curTime) ? frTime - curTime : curTime - frTime;
                                    if (diff > 5 * 60 * 1000) {
                                        //разница больше пяти минут
                                        cmdNum = TMCommand.commands.CM_GetDateTime;
                                        MessageBox.Create(getString(R.string.header_warning), "ВРЕМЯ ЧАСОВ КАССЫ И ВРЕМЯ КОНТРОЛЛЕРА ФН ОТЛИЧАЕТСЯ БОЛЕЕ ЧЕМ НА 5 МИНУТ\n\nЧАСЫ: " + Cdt + String.format(" (%d)",curTime) + "\nК.ФН: " + FRdt + String.format(" (%d)",frTime) + "\n\nЧТОБЫ НАСТРОИТЬ ВРЕМЯ 'ОК'", MessageBox.option.OK | MessageBox.option.CANCEL).show(getSupportFragmentManager(), MessageBox.TAG);
                                    }
                                    else {
                                        //запускаем проверку состояния ФН, только если он зарегистрирован
                                        if ( (cmd.GetHardStatus()&TMCommand.hard_status_flags.FN_REGISTERED) != 0 ) FRGetWaitDocs();
                                    }
                                } catch (Exception e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                        break;

                    case TMCommand.commands.CM_GetWaitDocs:
                        //получено количество документов на отправку
                        if ( (0xFFFF&cmd.GetFieldHexW(5)) > 0 ) {
                            //работа с ОФД
                            TMLib.getInstance().StartOFD();
                        }
                        break;

                    case TMCommand.commands.CM_ShiftReport:
                        //успешно выполнена
                        MessageBox.Create(getString(R.string.header_success), getString(R.string.msg_cmd_success)+"\n\nЗАБЕРИТЕ ОТЧЕТ",MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
                        break;

                    case TMCommand.commands.CM_ReceiptCancel:
                        //успешно выполнена
                        MessageBox.Create(getString(R.string.header_success), getString(R.string.msg_cmd_success)+"\n\nЧЕК ОТМЕНЕН",MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
                        break;

                }
            }
        }
    }
}
