package ru.nilsoft.example;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.os.Handler;
import androidx.appcompat.app.AppCompatActivity;
import android.os.Bundle;
import android.view.View;
import android.widget.TextView;

import java.io.UnsupportedEncodingException;

import ru.nilsoft.tm.TMCommand;
import ru.nilsoft.tm.TMError;
import ru.nilsoft.tm.TMLib;
import ru.nilsoft.tm.TMLibHandler;
import ru.nilsoft.tm.TMTag;

/**
 * Активность в которой собирается информация:
 * - о приложении;
 * - об оборудовании;
 * - о контроллере ФН;
 * - о текущем состоянии ФН.
 *
 * @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class InfoActivity extends AppCompatActivity {
    /** Handler для работы c контроллером ФН. */
    private Handler h;
    /** Номер выполняемой контроллером ФН команды. */
    private int cmdNum = 0;
    /** Статус обновления ключей **/
    private boolean fnSupportD7 = true;
    /** Уже выполнено. */
    private boolean isDone = false;
    /** Последний тег для команды получения параметров регистрации. */
    protected short regTLV = 0;
    /** Версия ФФД. */
    protected byte ffdVersion = 0;
    /** Зарегистрирован ли ФН для работы с маркировкой. */
    protected boolean isTMT = false;
    /** Тип сервера. */
    protected byte serverType = 0;
    /** UTC контроллера ФН, полученный от сервиса ФН. */
    private int fnUTC = 3; //поумолчанию часовой пояс +3 (Москва)

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_info);

        //создание хендлера
        h = new LibHandler();
    }

    @Override
    protected void onStart() {
        super.onStart();

        //регистрация хендлера для получения ответов от контроллера ФН
        TMLib.getInstance().registerHandler(h);

        if ( !isDone ) {
            //заполнение информационного поля
            isDone = true;
            //информация из Android
            AddText("Приложение: "+getString(R.string.app_name)+" "+BuildConfig.VERSION_NAME+String.format(" (%d)",BuildConfig.VERSION_CODE)+
                    "\nМодель: " + Build.MODEL +" Изготовитель: " + Build.MANUFACTURER +
                    "\nOS: "+ Build.DISPLAY + " Release: " + Build.VERSION.RELEASE +"\n");
            AddText("\n");
            //запрос на информацию из КФН
            TMLib.getInstance().SendRespVer();
        }
    }

    @Override
    protected void onStop() {
        //освобождение хендлера
        TMLib.getInstance().unregisterHandler(h);

        super.onStop();
    }

    /** Обработка нажатия кнопки ОК. */
    public void onButtonOK(View view) {
        finish();
    }

    /**
     * Получение заводского номера ККТ.
     */
    private void FRGetSerialNo() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetSerialNo();
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,2*CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получение информации о контроллее ФН.
     */
    private void FRGetInfo() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetInfo();
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,2*CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получение параметров регистрации ФН.
     * @param tlvId тэг параметра регистрации.
     */
    private void FRGetRegParam(short tlvId) {
        regTLV = tlvId;

        TMCommand cmd = new TMCommand();
        cmd.CmdFNGetRegParam(tlvId);
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получение даты/времени.
     */
    private void FRGetDateTime() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetDateTime();
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получить число уведомлений КМ.
     */
    private void FRNotifGetStatus() {
        TMCommand cmd = new TMCommand();
        cmd.CmdNotifGetStatus();
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получение статуса ФН.
     */
    private void FRFNGetStatus() {
        TMCommand cmd = new TMCommand();
        cmd.CmdFNGetStatus();
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получение времени последнего обновления ключей.
     */
    private void FRServiceFlagStatusKeys() {
        TMCommand cmd = new TMCommand();
        cmd.CmdServiceUpdateKeys(TMCommand.asokp_cmd.FLAG_UPDATE);
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получение времени последнего обновления ключей.
     */
    private void FRServiceUpdateKeys() {
        TMCommand cmd = new TMCommand();
        cmd.CmdServiceUpdateKeys(TMCommand.asokp_cmd.GET_LAST_UPDATE);
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получение параметров сервера.
     * @param type тип сервера.
     */
    private void FROFDGetParam(byte type) {
        TMCommand cmd = new TMCommand();
        if ( type == TMCommand.server_types.OFD ) cmd.CmdOFDGetParam(); //для совместимости с предыдущими версиями
        else cmd.CmdOFDGetParamExt(type);
        serverType = type;
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this,cmd,CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD));
    }

    /**
     * Получение информации об количестве неотправленных документов
     */
    public void FRGetWaitDocs() {
        TMCommand cmd = new TMCommand();
        cmd.CmdGetWaitDocs();
        cmdNum = cmd.GetCmdCode();
        TMLib.getInstance().DoCmd(this, cmd, CommonFunc.tmGetTimeOut(CommonFunc.tmTimeoutType.TM_TIMEOUT_CMD), false);
    }

    /** Вывод текста. */
    public void AddText(String text) {
        TextView viewText = findViewById(R.id.viewInfo);
        viewText.append(text);
    }

    /** Обработка нажатия кнопки ПЕЧАТЬ. */
    public void onButtonPrint(View view) {
        String data = String.valueOf(((TextView)findViewById(R.id.viewInfo)).getText());
        if (!data.isEmpty()) {
            byte[] tmp;
            try {
                //печать производится в CP866 формате.
                tmp = data.getBytes("Cp866");
            }
            catch (UnsupportedEncodingException e) {
                tmp = data.getBytes();
            }
            TMLib.getInstance().SendPrintData(tmp);
        }
    }

    /** Хендлер для обработки сообщений от TMLib. */
    class LibHandler extends TMLibHandler {

        /** Конструктор по умолчанию. */
        public LibHandler(){
            super(MainApp.GetCtx().getMainLooper());
        }

        @Override
        public void onRespCmd(TMCommand cmd) {
            //получен ответ на команду
            String text = "";
            boolean bErr = (cmd.GetMainError() != 0);

            //обработка ошибки

            if ( bErr ) {
                if( !( ( ((byte)cmd.GetCmdCode()) == TMCommand.commands.CM_ServiceUpdateKeys )
                    && ( cmd.GetMainError() == TMCommand.errors.ERR_UNKNOWN_FN_CMD) ) )
                text = String.format("КОМАНДА %02X НЕ ВЫПОЛНЕНА: ",cmd.GetCmdCode()) + TMError.GetText((byte)cmd.GetMainError()) + "\n";
            }

            switch ((byte) cmd.GetCmdCode()) {
                case TMCommand.commands.CM_GetSerialNo:
                    if ( !bErr ) {
                        text += "ККТ ЗН: " + cmd.GetFieldValue(5) + "\n";
                    }

                    //вызов следующей функции
                    FRGetInfo();
                    break;

                case TMCommand.commands.CM_GetInfo:
                    if ( !bErr ) {
                        text += "ТИП: " + cmd.GetFieldString(5) + "\n";
                        text += "РЕГ: " + cmd.GetFieldValue(8) + "\n";
                        text += "ИНН: " + cmd.GetFieldValue(9) + "\n";
                        text += "КОД АКТ: " + cmd.GetFieldString(12) + "\n";
                        text += "KKT HW: " + cmd.GetFieldString(6) + "\n";
                        //text += "ККТ FW: " + cmd.GetFieldString(10) + "\n";
                        text += "ККТ ПО: " + cmd.GetFieldString(10) + "\n";
                    }

                    if ( (cmd.GetHardStatus()&TMCommand.hard_status_flags.FN_REGISTERED) != 0 ) {
                        //вызов следующей функции
                        FRGetRegParam(TMTag.NUM_VERSION_FFD);
                    } else {
                        //если ФН незарегистрирован - сразу вызываем концовку
                        text += "ФН ЗН: " + cmd.GetFieldValue(11) + "\n";
                        text += GetStateKKT(cmd);
                    }
                    break;

                case TMCommand.commands.CM_FNGetRegParam:
                    if ( regTLV == TMTag.NUM_VERSION_FFD ) {
                        //получена версия ФФД по которому регистрирован ФН
                        if (!bErr) {
                            text += "РЕГИСТРАЦИЯ: ФФД ";
                            ffdVersion = cmd.GetFieldHexB(5);
                            switch (ffdVersion) {
                                case TMCommand.proto_ffd.FFD_1_05:
                                    text += "1.05\n";
                                    break;

                                case TMCommand.proto_ffd.FFD_1_1:
                                    text += "1.1\n";
                                    break;

                                case TMCommand.proto_ffd.FFD_1_2:
                                    text += "1.2";
                                    break;

                                default:
                                    text += "НЕИЗВЕСТЕН (" + cmd.GetFieldHexB(5) + ")\n";
                                    break;
                            }
                        }

                        //вызов следующей функции
                        if ( ffdVersion == TMCommand.proto_ffd.FFD_1_2 ) FRGetRegParam(TMTag.KKT_REG_FLAGS);
                        else FRGetDateTime();
                    }
                    else if ( regTLV == TMTag.KKT_REG_FLAGS ) {
                        //проверка, что зарегистрован поддержка ТМТ (поддержка маркированого товара)
                        if ( !bErr && ((cmd.GetFieldHexDW(5) & TMCommand.kkt_mode_reg.TMT) != 0) ) {
                            //есть поддержка маркированого товара
                            isTMT = true;
                            text += " TMT\n";
                        }
                        else {
                            //не поддерживается работа с маркировками
                            isTMT = false;
                            text += "\n";
                        }

                        //вызов следующей функции
                        FRGetDateTime();
                    }
                    break;

                case TMCommand.commands.CM_GetDateTime:
                    if ( !bErr ) {
                        text += "ККТ DT: " + cmd.GetFieldValue(5) + " " + cmd.GetFieldValue(6) + " UTC: " + fnUTC + "\n";
                    }

                    //вызов следующей функции
                    FRGetWaitDocs();
                    break;

                case TMCommand.commands.CM_GetWaitDocs:
                    if ( !bErr ) text += "ОФД ДОКУМЕНТЫ К ОТПРАВКЕ: " + cmd.GetFieldHexW(5) + "\n";

                    //вызов следующей функции
                    if ( isTMT ) FRNotifGetStatus();
                    else FRFNGetStatus();
                    break;

                case TMCommand.commands.CM_NotifGetStatus:
                    if ( !bErr ) text += "ОИСМ ДОКУМЕНТЫ К ОТПРАВКЕ: " + cmd.GetFieldHexW(6) + "\n";

                    //вызов следующей функции
                    FRServiceFlagStatusKeys();
                    break;

                case TMCommand.commands.CM_ServiceUpdateKeys:
                    if(fnSupportD7) {
                        if ( cmd.GetMainError() != TMCommand.errors.ERR_UNKNOWN_FN_CMD )
                        {
                            if ( !bErr ) {
                                byte keyFlag = cmd.GetFieldHexB(5);
                                switch (keyFlag) {
                                    case 0:
                                        text += "КЛЮЧИ АС ОКП ОБНОВЛЕНЫ.\n";
                                        break;
                                    case 1:
                                        text += "НЕОБХОДИМО ОБНОВИТЬ КЛЮЧИ АС ОКП.\n";
                                        break;
                                    case 2:
                                        text += "КЛЮЧИ АС ОКП НЕ ОБНОВЛЯЛИСЬ БОЛЕЕ 60 ДНЕЙ. ОБРАТИТЕСЬ В СЛУЖБУ ТЕХНИЧЕСКОЙ ПОДДЕРЖКИ.\n";
                                        break;
                                }
                            }
                            //вызов следующей функции
                            FRFNGetStatus();
                        }
                        else {
                            //команда D7 не поддерживается ФН'ом, проверяем по дате
                            fnSupportD7 = false;
                            FRServiceUpdateKeys();
                        }
                    }
                    else {
                        //команда D7 не поддерживается ФН'ом, проверяем по дате
                        if ( !bErr ) {
                            short keyDate = cmd.GetFieldHexW(5);
                            if (keyDate == 0) text += "НЕОБХОДИМО ОБНОВИТЬ КЛЮЧИ АС ОКП\n";
                            else {
                                text += "КЛЮЧИ АС ОКП ОБНОВЛЕНЫ " + String.format("%02d.%02d.%04d", keyDate & 0x1F, (keyDate >> 5) & 0x0F, 1996 + ((keyDate >> 9) & 0x7F)) + "\n";
                            }
                        }
                        //вызов следующей функции
                        FRFNGetStatus();
                    }

                    break;

                case TMCommand.commands.CM_FNGetStatus:
                    if ( !bErr ) {
                        text += "\nФН DT: " + cmd.GetFieldValue(10) + "\n";
                        text += "ФН ЗН: " + cmd.GetFieldValue(11) + "\n";
                        text += "СРОК ДЕЙСТВ.ФН: " + cmd.GetFieldValue(13) + "\n";
                        text += "ФН вер: " + cmd.GetFieldString(14) + "\n";
                        byte typeFN = cmd.GetFieldHexB(15);
                        text += "ФН тип: " + String.format("x%02X ", typeFN) + ((typeFN == 0) ? "отладочный" : (typeFN == 1) ? "боевой" : "не опр.") + "\n";
                        byte statFN = cmd.GetFieldHexB(5);
                        text += String.format(getString(R.string.fn_stage_title)+": x%02X\n", statFN);
                        if ((statFN & 0x0F) == 1) text += "- "+getString(R.string.fn_stage_1)+"\n";
                        else if ((statFN & 0x0F) == 3) text += "- "+getString(R.string.fn_stage_2)+"\n";
                        else if ((statFN & 0x0F) == 7) text += "- "+getString(R.string.fn_stage_3)+"\n";
                        else if ((statFN & 0x0F) == 15) text += "- "+getString(R.string.fn_stage_4)+"\n";
                        byte flagsFN = cmd.GetFieldHexB(9);
                        text += String.format("ФН предупреждения: x%02X\n", flagsFN);
                        if ((flagsFN & TMCommand.fn_warn_flags.NEED_CHANGE_FN) != 0)
                            text += "- до окончания срока действия 3 дня\n";
                        if ((flagsFN & TMCommand.fn_warn_flags.PREPARE_CHANGE_FN) != 0)
                            text += "- до окончания срока действия 30 дней\n";
                        if ((flagsFN & TMCommand.fn_warn_flags.MEMORY_FULL_FN) != 0)
                            text += "- архив ФН заполнен на 90%\n";
                        if ((flagsFN & TMCommand.fn_warn_flags.WAIT_RESP_OFD) != 0)
                            text += "- превышено время ожидания ответа ОФД\n";
                    }

                    //вызов следующей функции
                    FROFDGetParam(TMCommand.server_types.OFD);
                    break;

                case TMCommand.commands.CM_OFDGetParam:
                    byte type = serverType;

                    if( !bErr ) {
                        //имя сервера
                        String s = cmd.GetFieldString(9) + String.format(":%d", cmd.GetFieldHexW(5));

                        switch(type){
                            case TMCommand.server_types.OFD:
                                text += "\nОФД: " + s;
                                if ( isTMT ) {
                                    //вызов следующей функции
                                    FROFDGetParam(TMCommand.server_types.OISM);
                                }
                                break;

                            case TMCommand.server_types.OISM:
                                text += "\nОИСМ: " + s;
                                //вызов следующей функции
                                FROFDGetParam(TMCommand.server_types.ASOKP);
                                break;

                            case TMCommand.server_types.ASOKP:
                                text += "\nАС ОКП: " + s;
                                break;
                        }
                    }

                    if ( !isTMT || bErr || (type == TMCommand.server_types.ASOKP) ) text += "\n"+GetStateKKT(cmd);
                    break;
            }
            AddText(text);
        }

        /**
         * Получение и расшифровка состояния ККТ по ответу на команду.
         * @param cmd ответ команды.
         * @return расшифровка состояния.
         */
        public String GetStateKKT(TMCommand cmd) {
            String text;

            //состояние КФН
            int hardStatus = cmd.GetHardStatus();
            text = String.format("\nПОСТОЯННЫЙ СТАТУС: %02x\n", hardStatus);
            if ((hardStatus & TMCommand.hard_status_flags.HARDWARE_FAIL) != 0) text += "КОНТРОЛЛЕР ФН НЕИСПРАВЕН\n";
            if ((hardStatus & TMCommand.hard_status_flags.CTRL_MEMORY_FAIL) != 0) text += "СБОЙ КОНТРОЛЬНОЙ ПАМЯТИ\n";
            if ((hardStatus & TMCommand.hard_status_flags.FN_FAIL) != 0) text += "СБОЙ РАБОТЫ ФН\n";
            if ((hardStatus & TMCommand.hard_status_flags.FN_FULL) != 0) text += "ФН БЛИЗОК К ЗАПОЛНЕНИЮ\n";
            if ((hardStatus & TMCommand.hard_status_flags.FN_END) != 0) text += "ФН ЗАПОЛНЕН\n";
            if ((hardStatus & TMCommand.hard_status_flags.FN_REGISTERED) != 0) text += "ФН ЗАРЕГИСТРИРОВАН\n";
            else text += "ФН НЕ ЗАРЕГИСТРИРОВАН\n";

            int dynamicStatus = cmd.GetDynamicStatus();
            text += String.format("\nТЕКУЩИЙ СТАТУС: %04x\n", dynamicStatus);
            if ((dynamicStatus & TMCommand.dynamic_status_flags.FISCAL_DOC_MASK) != 0)
                text += "ОТКРЫТЫЙ ЧЕК\n";
            if ((dynamicStatus & TMCommand.dynamic_status_flags.SHIFT_NOT_CLOSE) != 0)
                text += "СМЕНА ОТКРЫТА\n";

            return text;
        }

        @Override
        public void onVersion(Bundle data, int hi_ver, int mi_ver, int lo_ver, int date_ver, String text) {
            //получена версия сервиса ККТ
            AddText("Сервис: "+text+"\n\n");
            fnUTC = data.getInt("utc",3); //по умолчанию делаем по МОСКВЕ

            //запускаем следующий запрос
            FRGetSerialNo();
        }

        @Override
        public void onTimeOut() {
            //сообщение о превышении таймаута обращения к контроллеру ФН
            AddText(String.format("\nСБОЙ ВЫПОЛНЕНИЯ КОМАНДЫ %02x\n",cmdNum)+"ПРЕВЫШЕН ТАЙМАУТ");
        }

        @Override
        public void onError(String errText) {
            //сообщение о сбое сервиса взаимодействия с контроллером ФН
            AddText("СБОЙ РАБОТЫ СЕРВИСА ВЗАИМОДЕЙСТВИЯ С КОНТРОЛЛЕРОМ: "+errText+"\n");
        }

        @Override
        public void onPrint(int mainErr, int subErr) {
            //получен ответ от модуля печати
            if (mainErr == 0) {
                //отрезать бумагу
                TMLib.getInstance().SendPrintCut();
                MessageBox.Create(getString(R.string.header_success), getString(R.string.msg_print_success), MessageBox.option.OK).show(getSupportFragmentManager(), MessageBox.TAG);
            }
            else {
                //сообщение об ошибке печати
                MessageBox.Create(getString(R.string.header_error), String.format(getString(R.string.msg_print_error), mainErr, subErr)+"\n\n"+TMError.GetPrnText(mainErr, subErr), MessageBox.option.OK | MessageBox.option.WARN).show(getSupportFragmentManager(), MessageBox.TAG);
            }
        }
    }
}
