package ru.nilsoft.example;

import android.text.format.DateFormat;
import android.util.Log;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.util.Calendar;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import ru.nilsoft.tm.TMDefaults;

/**
 * Класс содержит вспомогательные функции для работы.
 *
 * @author <a href="http://www.nilsoft.ru">www.nilsoft.ru</a>, <a href="mailto:nilstarsoft@mail.ru">nilstarsoft@mail.ru</a>
 */
public class CommonFunc {
    /** Количество знаков после запятой (копеек). */
    static final int CURRENCY_RADIX = 2;
    /** Нулевое значение валюты. */
    static final String CURRENCY_ZERO = "0.00";
    /** Минимальное значение валюты (1 копейка). */
    static final String CURRENCY_MIN = "0.01";
    /** Максимальная сумма для демо-лицензии. */
    static final String CURRENCY_MAX_DEMO = "100.00";
    /** Максимальная длина стоимости товара. */
    static final int CURRENCY_MAX_PRICE_LEN = 14;

    /**
     * Проверка на ноль.
     * @return false: значение не нулевое, true: нулевое.
     * @param s [in] - цифровая строка.
     */
    public static boolean ValueIsZero(String s) {
        try {
            return (Long.parseLong(s.replace(".","")) == 0);
        }
        catch(NumberFormatException e){
            //e.printStackTrace();
        }
        return true; //если на входе шлак, то считаем что нулевое значение
    }

    /**
     * Проверка что строка это число.
     * @return false: значение не число, true: число.
     * @param s [in] - строка.
     */
    public static boolean ValueIsNum(String s) {
        if ( s != null ) {
            for(int i=0; i<s.length(); i++) {
                char c = s.charAt(i);
                if ( (c < '0') || (c > '9') ) return false;
            }
            return true;
        }
        return false;
    }

    /**
     * Преобразование значения с требуемым количеством точек после запятой.
     * @return результат.
     * @param s - исходная цифровая строка.
     * @param radix - требуемое количество разрядов после запятой.
     */
    public static String ValueRound(String s, int radix) {
        String ret = s;
        if ( (ret == null) || ret.isEmpty() ) ret = "0"; //по умолчанию == 0
        boolean isNegative = (ret.charAt(0)=='-');
        int rdx = ret.indexOf('.');

        if ( rdx < 0 ) {
            //точка не найдена в числе
            rdx = 0;
        }
        else {
            //убираем точку
            ret = RemoveDot(ret);
            rdx = ret.length() - rdx;
        }

        if ( rdx == radix ) {
            //кол-во разрядов после запятой совпадает: ничего не делаем
            return s;
        }
        else if ( rdx < radix ) {
            //кол-во разрядов после запятой у значения меньше требуемого: добиваем нулями
            for ( ; rdx < radix; rdx++ ) ret = ret.concat("0");
        }
        else {
            //кол-во разрядов после запятой у значения больше требуемого: обрезаем число
            //(необходимо делать округление)
            boolean isCarry = (ret.charAt(ret.length()-(rdx-radix)) >= '5');
            ret = ret.substring(0,ret.length()-(rdx-radix));
            if ( isCarry ) {
                try {
                    long l = Long.parseLong(ret);
                    if ( isNegative ) l--; else l++;
                    ret = Long.toString(l);
                } catch (NumberFormatException e) {
                    e.printStackTrace();
                }
            }
        }

        if ( radix > 0 ) {
            //добавляем точку
            ret = SetDot(ret, radix);
        }

        //Log.d("ValueRound",s+" radix "+radix+" -> "+ret);
        return ret;
    }

    /**
     * Сравнение цифровых строк.
     * @return <0: s1<s2, 0: s1==s2, >0: s1>s2.
     * @param s1 - первая цифровая строка.
     * @param s2 - вторая цифровая строка.
     */
    public static int ValuesCompare(String s1, String s2) {
        int rdx1 = s1.indexOf('.');
        int rdx2 = s2.indexOf('.');

        if ( rdx1 < 0 ) {
            //точка не найдена в числе
            rdx1 = 0;
        }
        else {
            //убираем точку
            s1 = RemoveDot(s1);
            rdx1 = s1.length()-rdx1;
        }

        if ( rdx2 < 0 ) {
            //точка не найдена в числе
            rdx2 = 0;
        }
        else {
            //убираем точку
            s2 = RemoveDot(s2);
            rdx2 = s2.length()-rdx2;
        }

        //выравнивание чисел
        if ( rdx2 > rdx1 ) for( ; rdx2 > rdx1; rdx1++ ) s1 = s1.concat("0");
        if ( rdx1 > rdx2 ) for( ; rdx1 > rdx2; rdx2++ ) s2 = s2.concat("0");

        //Log.d("ValuesCompare","s1="+s1+" s2="+s2);
        try {
            //сравнение
            long l1 = Long.parseLong(s1);
            long l2 = Long.parseLong(s2);

            if (l1 < l2) return -1;
            if (l1 > l2) return 1;
        }
        catch(NumberFormatException e){
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * Сложение числовых строк
     * (результат имеет столько же разрядов после запятой как у первого слагаемого).
     * @return результат (s1 + s2).
     * @param s1 - первое слагаемое.
     * @param s2 - второе слагаемое.
     */
    public static String ValuesAdd(String s1, String s2) {
        String ret = "0";
        int rdx = s1.indexOf('.');

        if ( rdx < 0 ) {
            //точка не найдена в числе
            rdx = 0;
        }
        else {
            //убираем точку
            s1 = RemoveDot(s1);
            rdx = s1.length()-rdx;
        }

        //подготовка второго слагаемого
        s2 = ValueRound(s2,rdx);
        if ( rdx > 0 ) s2 = RemoveDot(s2);

        try {
            //сложение
            long l = Long.parseLong(s1)+Long.parseLong(s2);
            ret = Long.toString(l);
        }
        catch(NumberFormatException e){
            e.printStackTrace();
        }

        if ( rdx > 0 ) {
            ret = SetDot(ret,rdx);
        }

        //Log.d("ValuesAdd",s1+" + "+s2+" = "+ret);
        return ret;
    }

    /**
     * Вычитание числовых строк
     * (результат имеет столько же разрядов после запятой как у первого числа).
     * @return результат (s1 - s2).
     * @param s1 - из чего вычитают.
     * @param s2 - что вычитают.
     */
    public static String ValuesSub(String s1, String s2) {
        String ret = "0";
        int rdx = s1.indexOf('.');

        if ( rdx < 0 ) {
            //точка не найдена в числе
            rdx = 0;
        }
        else {
            //убираем точку
            s1 = RemoveDot(s1);
            rdx = s1.length()-rdx;
        }

        //подготовка вычитаемого
        s2 = ValueRound(s2,rdx);
        if ( rdx > 0 ) s2 = RemoveDot(s2);

        try {
            //сложение
            long l = Long.parseLong(s1)-Long.parseLong(s2);
            ret = Long.toString(l);
        }
        catch(NumberFormatException e){
            e.printStackTrace();
        }

        if ( rdx > 0 ) {
            ret = SetDot(ret,rdx);
        }

        //Log.d("ValuesSub",s1+" - "+s2+" = "+ret);
        return ret;
    }

    /**
     * Перемножение числовых строк.
     * @return результат (s1 * s2).
     * @param s1 - первый множитель.
     * @param s2 - второй множитель.
     */
    public static String ValuesMul(String s1, String s2) {
        String ret = "0";
        int rdx1 = s1.indexOf('.');
        int rdx2 = s2.indexOf('.');

        if ( rdx1 < 0 ) {
            //точка не найдена в числе
            rdx1 = 0;
        }
        else {
            //убираем точку
            s1 = RemoveDot(s1);
            rdx1 = s1.length()-rdx1;
        }

        if ( rdx2 < 0 ) {
            //точка не найдена в числе
            rdx2 = 0;
        }
        else {
            //убираем точку
            s2 = RemoveDot(s2);
            rdx2 = s2.length()-rdx2;
        }

        try {
            //умножение
            long l = Long.parseLong(s1)*Long.parseLong(s2);
            ret = Long.toString(l);
        }
        catch(NumberFormatException e){
            e.printStackTrace();
        }

        if ( rdx1+rdx2 > 0 ) {
            ret = SetDot(ret,rdx1+rdx2);
        }

        //Log.d("ValuesMul",s1+" * "+s2+" = "+ret);
        return ret;
    }

    /**
     * Получение значение процента от числа
     * (результат имеет столько же разрядов после запятой как у первого числа).
     * @return результат.
     * @param s - значение от которого вычисляется процент.
     * @param percent - значение процента (0..100).
     */
    public static String ValuesPercent(String s, String percent) {
        String ret = "0";
        int rdxs = s.indexOf('.');
        int rdxp = percent.indexOf('.');

        if ( rdxs < 0 ) {
            //точка не найдена в числе
            rdxs = 0;
        }
        else {
            //убираем точку
            s = RemoveDot(s);
            rdxs = s.length()-rdxs;
        }

        if ( rdxp < 0 ) {
            //точка не найдена в числе
            rdxp = 0;
        }
        else {
            //убираем точку
            percent = RemoveDot(percent);
            rdxp = percent.length()-rdxp;
        }

        try {
            //вычисление процента
            long l = Long.parseLong(s)*Long.parseLong(percent);
            l = (l/100)+(((l%100)>=50)?1:0);
            while( rdxp > 0 ) {
                //убираем влияние разрядов после запятой у процента
                l = (l/10)+(((l%10)>=5)?1:0);
                rdxp--;
            }
            ret = Long.toString(l);
        }
        catch(NumberFormatException e){
            e.printStackTrace();
        }

        if ( rdxs > 0 ) {
            ret = SetDot(ret,rdxs);
        }

        //Log.d("ValuesPercent",s+" of "+percent+"% is "+ret);
        return ret;
    }

    /**
     * Получение остатка от деления
     * (результат имеет столько же разрядов после запятой как у первого числа).
     * @return результат остаток от деления (s1 % s2).
     * @param s1 [in] - делимое.
     * @param s2 [in] - делитель.
     */
    public static String ValuesRem(String s1, String s2) {
        String ret = "0";
        int rdx1 = s1.indexOf('.');
        int rdx2 = s2.indexOf('.');

        if ( rdx1 < 0 ) {
            //точка не найдена в числе
            rdx1 = 0;
        }
        else {
            //убираем точку
            s1 = RemoveDot(s1);
            rdx1 = s1.length()-rdx1;
        }

        if ( rdx2 < 0 ) {
            //точка не найдена в числе
            rdx2 = 0;
        }
        else {
            //убираем точку
            s2 = RemoveDot(s2);
            rdx2 = s2.length()-rdx2;
        }

        //выравнивание чисел
        int rdx = Math.max(rdx2, rdx1);
        if ( rdx > rdx1 ) for( int r = rdx1; rdx > r; r++ ) s1 = s1.concat("0");
        if ( rdx > rdx2 ) for( ; rdx > rdx2; rdx2++ ) s2 = s2.concat("0");

        try {
            //вычисление остатка
            long l = Long.parseLong(s1)%Long.parseLong(s2);
            ret = Long.toString(l);
        }
        catch(NumberFormatException e){
            e.printStackTrace();
        }

        if ( rdx > 0 ) {
            ret = SetDot(ret,rdx);
            //округление до количества разрядов после запятой у значения
            if ( rdx > rdx1 ) {
                ret = ValueRound(ret,rdx1);
            }
        }

        //Log.d("ValuesRem",s1+" rem "+s2+" = "+ret);
        return ret;
    }

    /**
     * Расчет скидки: Расчет скидки на позицию с использованием пропорции.
     * @param price текущая цена позиции [формат "<Р>.КК"].
     * @param origPrice оригинальная цена позиции [формат "<Р>.КК"] (используется для расчета процента).
     * @param maxprc максимальный процент скидки от оригинальной цены.
     * @param amount целевая сумма чека [формат "<Р>.КК"] (для расчета пропорции).
     * @param total текущая сумма чека [формат "<Р>.КК"] (для расчета пропорции).
     * @return новая цена позиции [формат "<Р>.КК"].
     */
    public static String CalcDiscount(String price, String origPrice, String maxprc, String amount, String total) {
        //текущая цена (в копейках)
        long iprice = Long.parseLong(RemoveDot(price));
        //оригинальная цена (в копейках)
        long iorig = Long.parseLong(RemoveDot(origPrice));
        //процент (в сотых долях процента, т.е. 10000 = 100%)
        long iprc = Long.parseLong(RemoveDot(ValueRound(maxprc,2)));
        //целевая сумма чека (в копейках)
        long iamount = Long.parseLong(RemoveDot(amount));
        //текущая сумма чека (в копейках)
        long itotal = Long.parseLong(RemoveDot(total));

        //расчет новой цены позиции по пропорции
        long inewprice = (iprice*iamount)/itotal;
        //расчет процента от оригинальной цены позиции
        long iprcprice = ((iorig*(10000-iprc))/10000)+((((iorig*(10000-iprc))%10000)>=5000)?1:0);

        if ( BuildConfig.DEBUG ) Log.d("CalcDiscount",String.format(": price=%d (%d) %sprice=%d newprice=%d",iprice,iorig,maxprc,iprcprice,inewprice));

        //скидка по позиции не должна превышать максимальный процент
        if ( inewprice < iprcprice ) inewprice = iprcprice;

        //преобразуем новую цену в строковый вид
        String rez = Long.toString(inewprice);

        //добавление точки
        return SetDot(rez,CURRENCY_RADIX);
    }

    /**
     * Расчет скидки: Корректирование цены (добавление сколько влезет из остатка суммы со скидкой).
     * @param price текущая цена позиции [формат "<Р>.КК"] (после применения Common_CalcDiscount).
     * @param origPrice оригинальная цена позиции [формат "<Р>.КК"].
     * @param qty количество товара позиции.
     * @param amount остаток суммы со скидкой по чеку [формат "<Р>.КК"] (значение на которую надо увеличить сумму позиции).
     * @return новая цена позиции [формат "<Р>.КК"].
     */
    public static String CalcDiscountCorrect(String price, String origPrice, String qty, String amount) {
        int rdx = 0;
        if ( qty.lastIndexOf('.') >= 0 ) {
            //удаление завершающих нулей
            while(qty.charAt(qty.length()-1) == '0') {
                qty = qty.substring(0,qty.length()-1);
            }
            //количесто разрядов после запятой
            rdx = qty.lastIndexOf('.');
            qty = RemoveDot(qty);
            rdx = qty.length()-rdx;
        }

        //текущая цена
        long iprice = Long.parseLong(RemoveDot(price));
        //оригинальная цена
        long iorig = Long.parseLong(RemoveDot(origPrice));
        //значение на которую надо увеличить сумму позиции
        long iamount = Long.parseLong(RemoveDot(amount));
        //количество
        long iqty = Long.parseLong(qty);

        //учет количества разрядов количества
        if ( rdx > 0 ) {
            for(int i=0; i<rdx; i++) {
                iamount = iamount*10;
            }
        }

        //расчет значения увеличения цены
        long icorrect = iamount/iqty;
        if ( rdx > 0 ) {
            //учет знаков после запятой у количества
            for(int i=0; i<rdx; i++) {
                icorrect = icorrect/10;
            }
        }

         //получение новой цены
        iprice = iprice+icorrect;
        if ( iprice > iorig ) {
            //новая цена не может превышать оригинальную цену
            iprice = iorig;
        }

        if( BuildConfig.DEBUG ) Log.d("CalcDiscountCorrect",String.format(": icorrect=%d iamount=%d iqty=%d iprice=%d iorig=%d",icorrect,iamount,iqty,iprice,iorig));

        String rez = Long.toString(iprice);

        //добавление точки
        return SetDot(rez,CURRENCY_RADIX);
    }

    /**
     * Расчет скидки: Корректирование количества товара (расчет с разделением позиции).
     * Расчет происходит следующим образом:
     * - подразумевается, что количество товара может делится по младшему разряду
     *   (т.е. 5 может поделиться на 3 и 2, а 0.100 разделиться на 0.055 и 0.045);
     * - остаток цены по сути преобразуется в количество товара;
     * - порядок изменения стоимости зависит от остатка цены и текущего количества товара.
     * @param price [in] - текущая цена (после применения Common_CalcDiscountCorrect).
     * @param origPrice [in] - исходная/оригинальная цена [формат: "<Р>.KK"].
     * @param qty [in] - количество товара (рекомендуется не более трех разрядов после запятой).
     * @param amount [in] - остаток суммы на которую надо увеличить стоимость товара [формат: "<Р>.KK"].
     * @return null: неуспешно
     *         [0]: новая цена товара (для новой позиции) [формат: "<Р>.KK"].
     *         [1]: новое кол-во товара (для новой позиции).
     */
    public static String [] CalcDiscountQtyCorrect(String price, String origPrice, String qty, String amount) {
        int rdx = 0;
        if ( qty.lastIndexOf('.') >= 0 ) {
            //удаление завершающих нулей
            while(qty.charAt(qty.length()-1) == '0') {
                qty = qty.substring(0,qty.length()-1);
            }
            //количесто разрядов после запятой
            rdx = qty.lastIndexOf('.');
            qty = RemoveDot(qty);
            rdx = qty.length()-rdx;
        }

        //текущее количество товара
        long iqty = Long.parseLong(qty);
        //остаток суммы, на которую надо увеличить стоимость
        long iamount = Long.parseLong(RemoveDot(amount));
        //текущая цена позиции товара (в копейках)
        long iprice = Long.parseLong(RemoveDot(price));
        //оригинальная цена позиции товара (в копейках)
        long iorig = Long.parseLong(RemoveDot(origPrice));
        //минимальное увеличение стоимости товара (1 копейка)
        long imin = 1;

        if ( rdx > 0 ) {
            //учет разрядов после запятой у количества товара
            //qty    0.55        qty     55
            //price  100    =>  price 10000
            //amount 004        amount  400
            for(int i=0; i<rdx; i++) {
                iprice  = iprice*10;
                iorig   = iorig*10;
                imin    = imin*10;

                //по сути далее преобразуется [остаток суммы] в [новое количество]
                if ( iamount > iqty ) {
                    //если изменение стоимости больше чем количество, увеличиваем минимальный прирост цены на порядок
                    imin = imin*10;
                } else {
                    //если изменение стоимости меньше чем количество, увеличиваем остаток
                    iamount = iamount*10;
                }
            }
        }

        if ( BuildConfig.DEBUG ) Log.d("CalcDiscountQtyCorrect",String.format(": price=%d(%d) amount=%d min=%d qty=%d rdx=%d\n",iprice,iorig,iamount,imin,iqty,rdx));

        //эту позицию нельзя расщепить, так как [новое количество] > [кол-во товара]
        if ( iamount > iqty ) return null;

        //новая цена (не должна превышать оригинальную)
        iprice = iprice+imin;
        if ( iprice > iorig ) return null;

        //учет знаков после запятой у количества
        if ( rdx > 0 ) {
            for(int i=0; i<rdx; i++) {
                iprice  = iprice/10;
            }
        }

        String [] rez = new String [2];

        //получение строки новой цены
        rez[0] = SetDot(Long.toString(iprice),CURRENCY_RADIX);

        //получение строки нового количества
        if ( rdx > 0 ) rez[1] = SetDot(Long.toString(iamount),rdx);
        else rez[1] = Long.toString(iamount);

        if ( BuildConfig.DEBUG ) Log.d("CalcDiscountQtyCorrect",String.format(": price=%d[%s] amount=%d[%s]",iprice,rez[0],iamount,rez[1]));

        return rez;
    }

    /**
     * Изъятие точки из строки.
     * @param value строка с точкой.
     * @return строка без точки.
     */
    public static String RemoveDot(String value) {
        if ( value == null ) return "";

        return value.replace(".","");
    }

    /**
     * Добавление точки в строку.
     * @param value строка числа.
     * @param radix количество символов после запятой.
     * @return число с точкой.
     */
    public static String SetDot(String value, int radix) {
        if ( value == null ) return "";


        if ( radix > 0 ) {
            int sign = ( value.charAt(0) == '-' )?1:0;

            //добавляем нули если не хватает
            StringBuilder v = new StringBuilder(value);
            while (v.length()-sign < radix+1) v.insert(0, "0");

            //добавляем точку
            v.insert(v.length()-(radix+sign),".");


            value = v.toString();
        }

        return value;
    }

    /**
     * Обрезание строки от неиспользуемых символов.
     * @param s строка
     * @return обрезанная строка
     */
    public static String Trim(String s) {
        if ( s != null ) {
            //вначале строки
            while ((!s.isEmpty()) && (s.charAt(0) <= 0x20)) {
                s = s.substring(1);
            }
            //вконце строки
            while ((!s.isEmpty()) && (s.charAt(s.length() - 1) <= 0x20)) {
                s = s.substring(0, s.length() - 1);
            }
        }
        return s;
    }

    /**
     * Конвертирование байтов в HEX-представление.
     *
     * @param src    буфер с исходными данными.
     * @param srcLen длина данных.
     * @return HEX-представление.
     */
    public static byte[] BytesToHex(byte[] src, int srcLen) {
        byte[] dst = new byte[srcLen * 2];
        for (int i = 0; i < srcLen; i++) {
            byte b = src[i];
            if ((b & 0x0F) < 10) {
                dst[(2 * i) + 1] = (byte) ('0' + (b & 0x0F));
            } else {
                dst[(2 * i) + 1] = (byte) ('A' + (b & 0x0F) - 10);
            }
            b = (byte) (b >> 4);
            if ((b & 0x0F) < 10) {
                dst[2 * i] = (byte) ('0' + (b & 0x0F));
            } else {
                dst[2 * i] = (byte) ('A' + (b & 0x0F) - 10);
            }
        }
        return dst;
    }

    /**
     * Конвертирование шестанадцатеричной строки в байтовый массив.
     *
     * @param src    исходный буфер (шестнадцатеричная строка).
     * @param srcLen длина данных для конвертации.
     * @return байтовый массив.
     */
    public static byte[] HexToBytes(byte[] src, int srcLen) {
        byte[] dst = new byte[(srcLen / 2) + (((srcLen & 1) == 0) ? 0 : 1)];
        for (int i = 0; i < srcLen; i++) {
            if ((i & 1) == 0) {
                int b = ((src[i] >= '0') && (src[i] <= '9')) ? (src[i] - '0') :
                        ((src[i] >= 'A') && (src[i] <= 'F')) ? (src[i] - 'A' + 10) :
                                ((src[i] >= 'a') && (src[i] <= 'f')) ? (src[i] - 'a' + 10) : 0;
                dst[i / 2] = (byte) (0xFF & (b << 4));
            } else {
                int b = (((src[i] >= '0') && (src[i] <= '9')) ? (src[i] - '0') :
                        ((src[i] >= 'A') && (src[i] <= 'F')) ? (src[i] - 'A' + 10) :
                                ((src[i] >= 'a') && (src[i] <= 'f')) ? (src[i] - 'a' + 10) : 0) + (0xFF & dst[i / 2]);
                dst[i / 2] = (byte) (0xFF & b);
            }
        }
        return dst;
    }

    /**
     * Распаковка zip файла.
     * @param zipFile файл для распаковки.
     * @param targetDirectory куда распаковать
     * @throws IOException исключение распаковки.
     */
    public static void UnzipFile(File zipFile, File targetDirectory) throws IOException {
        //long total_len = zipFile.length();
        //long total_installed_len = 0;

        try (ZipInputStream zis = new ZipInputStream(new BufferedInputStream(new FileInputStream(zipFile)))) {
            ZipEntry ze;
            int count;
            byte[] buffer = new byte[1024];
            while ((ze = zis.getNextEntry()) != null) {
                File file = new File(targetDirectory, ze.getName());
                File dir = ze.isDirectory() ? file : file.getParentFile();
                if (!dir.isDirectory() && !dir.mkdirs())
                    throw new FileNotFoundException("Failed to ensure directory: " + dir.getAbsolutePath());
                if (ze.isDirectory()) continue;
                FileOutputStream fout = new FileOutputStream(file);
                try {
                    while ((count = zis.read(buffer)) != -1) fout.write(buffer, 0, count);
                } finally {
                    fout.close();
                }

                // if time should be restored as well
                long time = ze.getTime();
                if (time > 0) file.setLastModified(time);
            }
        }
    }

    /**
     * Упаковка файла в архив.
     * @param file файл для упаковки.
     * @param fileName название в архиве.
     * @param zipFile результирующий файл.
     * @throws IOException исключение упаковки.
     */
    public static void ZipFile(File file, String fileName, File zipFile) throws IOException {
        try ( ZipOutputStream out = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(zipFile)))) {
            byte[] buffer = new byte[1024];

            try(BufferedInputStream origin = new BufferedInputStream(new FileInputStream(file), buffer.length) ) {
                ZipEntry entry = new ZipEntry(fileName);
                out.putNextEntry(entry);
                int count;

                while ((count = origin.read(buffer, 0, buffer.length)) != -1) {
                    out.write(buffer, 0, count);
                }
            }
        }
    }

    /**
     * Преобразование строкового числа в int.
     * @param s строковое число.
     * @return число.
     */
    public static int StrToInt(String s) {
        int ret = 0;
        int sign = 1;

        if ( ( s == null) || s.isEmpty() ) return 0;

        //определяем знак числа
        if ( s.charAt(0) == '-' ) {
            //отрицательное число
            sign = -1;
            s = s.substring(1);
            if (s.isEmpty()) return 0;
        }

        if ( s.startsWith("0x") || s.startsWith("0X") || ( s.charAt(0)=='x' ) || ( s.charAt(0)=='X' ) ) {
            //hex value
            int mul = 1;
            int len = (s.charAt(0)=='0')?1:0;

            //length calculate
            while( ( len+1 < s.length() ) &&
                   ( ( ( s.charAt(len+1) >= '0' ) && ( s.charAt(len+1) <= '9' ) ) ||
                     ( ( s.charAt(len+1) >= 'a' ) && ( s.charAt(len+1) <= 'h' ) ) ||
                     ( ( s.charAt(len+1) >= 'A' ) && ( s.charAt(len+1) <= 'H' ) ) ) ) len++;

            //get value
            for(int i=0; i<len; i++) {
                char b = s.charAt(len-i);
                if ( ( b >= '0' ) && ( b <= '9' ) ) {
                    ret += (b-'0') *mul;
                }
                else if ( ( b >= 'A') && ( b <= 'H' ) ) {
                    ret += (10+(b-'A')) *mul;
                }
                else if ( ( b >= 'a' ) && ( b <= 'h' ) ) {
                    ret += (10+(b-'a')) *mul;
                }
                else break;
                mul *= 16;
            }
        }
        else {
            //dec value
            int mul = 1;
            int len = 0;

            //length calculate
            while( ( len < s.length() ) && ( s.charAt(len) >= '0' ) && ( s.charAt(len) <= '9' ) ) len++;

            //get value
            for(int i=0; i<len; i++) {
                char b = s.charAt(len-(i+1));
                if ( ( b >= '0' ) && ( b <= '9' ) ) {
                    ret += (b-'0') *mul;
                }
                mul *= 10;
            }
        }

        return ret*sign;
    }

    /**
     * Вычисление UPC контрольного символа.
     * @param code штрихкод.
     * @param len длина значимых символов.
     * @return контрольный символ.
     */
    public static char BarCodeUPCCtrl(String code, int len)
    {
        int k = 0;
        int i;
        for(i=1; i<=len; i++) {
            k += (code.charAt(len-i)-'0')*(((i&1)!=0)?3:1);
        }
        return (char)('0'+((10-(k%10))%10));
    }

    /**
     * Сравнение штрихкодов.
     * @param code1 первый штрихкод.
     * @param code2 второй штрихкод.
     * @return true: коды эквивалентны, false: разные.
     */
    public static boolean BarCodeIsEqual(String code1, String code2) {
        if ( ( code1 != null ) && ( code2 != null ) ) {
            int len1 = code1.length();
            int len2 = code2.length();
            String codeShort = (len1 < len2) ? code1 : code2;
            String codeLong = (len1 < len2) ? code2 : code1;
            int lenShort = Math.min(len1, len2);
            int lenLong = Math.max(len1, len2);

            if (lenShort == lenLong) {
                //проверка что коды одинаковые
                if (codeShort.equals(codeLong)) return true;
            }
            else if (lenShort+1 == lenLong) {
                //UPC code (сканер вернул значение без котрольного кода или последней цифры)
                if ( codeLong.startsWith(codeShort) && (BarCodeUPCCtrl(codeShort, lenShort) == codeLong.charAt(lenShort)))
                    return true;

                //EAN code начинанается с '0' интерпретируется сканером как UPC code
                if ( (codeLong.charAt(0) == '0') && codeLong.startsWith(codeShort,1) )
                    return true;
            }
    		else if ( ( lenShort+2 == lenLong ) &&
                      ( codeLong.charAt(0) == '0' ) &&
                      codeLong.startsWith(codeShort,1) &&
                      ( BarCodeUPCCtrl(codeShort, lenShort) == codeLong.charAt(lenShort+1) ) ) {
                //EAN code начинанается с '0' интерпретируется сканером как UPC code
                //UPC code (сканер вернул значение без котрольного кода или последней цифры)
                return true;
            }
    		else if ( ( codeLong.charAt(lenShort) == 0x1D ) && codeLong.startsWith(codeShort) )
            {
                //маркировочный код может сравниваться с обрезанным маркировочным кодом
                return true;
            }
        }
        return false;
    }

    /**
     * Получение времени ввиде форматированной строки.
     * @param cal объект календаря.
     * @return строка.
     */
    public static String GetFormattedDateTime(Calendar cal) {
        return DateFormat.format("dd-MM-yyyy HH:mm:ss", cal).toString();
    }

    /**
     * Получение текущего времени ввиде форматированной строки.
     * @return строка.
     */
    public static String GetFormattedDateTime() {
        return GetFormattedDateTime(Calendar.getInstance());
    }

    /**
     * Преобразование документа в строку.
     *
     * @return строка.
     * @param node преобразуемая нода.
     */
    public static String StringFromNode(Node node) {
        try {
            DOMSource domSource = new DOMSource(node);
            StringWriter writer = new StringWriter();
            StreamResult result = new StreamResult(writer);
            TransformerFactory tf = TransformerFactory.newInstance();
            Transformer transformer = tf.newTransformer();
            transformer.transform(domSource, result);
            return writer.toString();
        }
        catch(TransformerException e) {
            if (BuildConfig.DEBUG) e.printStackTrace();
            return "";
        }
    }

    /**
     * Поиск элемента по пути в xml.
     *
     * @param root раздел, начиная от которого идет поиск.
     * @param path путь к искомому разделу с использованием разделителя '/'.
     * @return найденый подраздел, null: подраздел не найден.
     */
    public static Element GetElement(final Node root, final String path) {
        if ( ( root != null ) && ( path != null) ) {
            try {
                Node var = (Node) root;
                //разбиваем на подстроки
                for (String itemName : path.split("/")) {
                    //ищем элемент соответствующий подстроке
                    var = var.getFirstChild();
                    while (var != null) {
                        String s = var.getNodeName();
                        if ((s != null) && (s.equals(itemName)) && (var instanceof Element)) {
                            //найден раздел
                            break;
                        }
                        var = var.getNextSibling();
                    }
                    if (var == null) break;
                }
                return (Element) var;
            } catch (Exception e) {
                if (BuildConfig.DEBUG) e.printStackTrace();
            }
        }
        return null;
    }

    /**
     * Получение атрибута элемента XML.
     * @param root раздел, начиная от которого идет поиск.
     * @param path путь к искомому разделу с использованием разделителя '/'.
     * @param attr название атрибута.
     * @return "": атрибут не науден, иначе строка атрибута.
     */
    public static String GetElementAttr(final Node root, final String path, final String attr) {
        Element element = GetElement(root,path);
        if ( element != null ) return element.getAttribute(attr);
        return "";
    }

    /**
     * Установка атрибута элемента XML (елемент должен существовать).
     * @param root раздел, начиная от которого идет поиск.
     * @param path путь к искомому разделу с использованием разделителя '/'.
     * @param attr название атрибута.
     * @param value значение атрибута.
     * @return false: элемент не найден.
     */
    public static boolean SetElementAttr(final Node root, final String path, final String attr, final String value) {
        Element element = GetElement(root,path);
        if ( element != null ) {
            element.setAttribute(attr, value);
            return true;
        }
        return false;
    }

    /**
     * Копирование атрибута по имени из одного елемента в другой.
     * @param src источник.
     * @param dst приемник.
     * @param name название атрибута.
     * @return true: скопировано, false: нет атрибута у источника.
     */
    public static boolean CopyElementAttr(final Element src, final Element dst, final String name) {
        if ( ( src != null ) && ( dst != null ) ) {
            //считываем атрибут
            String value = src.getAttribute(name);
            if (value.length() > 0) {
                //сохранение в dst
                dst.setAttribute(name, value);
                return true;
            } else dst.removeAttribute(name);
        }
        return false;
    }

    /**
     * Копирование всех атрибутов из одного элемента в другой.
     * @param src источник.
     * @param dst приемник.
     * @return true: скопировано, false: нет атрибутов у источника.
     */
    public static boolean CopyElementAttr(final Node src, final Element dst) {
        if ( ( src != null ) && ( dst != null ) ) {
            //чтение всех атрибутов из src
            NamedNodeMap nodeMap = src.getAttributes();
            if ( nodeMap.getLength() > 0 ) {
                //сохранение в dst
                for (int i = 0; i < nodeMap.getLength(); i++) {
                    Node n = nodeMap.item(i);
                    dst.setAttribute(n.getNodeName(), n.getNodeValue());
                }
                return true;
            }
        }
        return false;
    }

    /**
     * Сохранение XML структуры в файл.
     *
     * @param f файл для сохранения.
     * @param root документ для сохранения.
     * @return 0: успешно, иначе ошибка.
     */
    public static boolean SaveXML(final File f, final Document root) {
        if ( root != null ) {
            try (FileOutputStream fos = new FileOutputStream(f)) {
                TransformerFactory.newInstance().newTransformer().transform(new DOMSource(root), new StreamResult(fos));
                return true;
            } catch (Exception e) {
                if (BuildConfig.DEBUG) e.printStackTrace();
            }
        }
        return false;
    }

    /**
     * Загрузка XML из файла.
     *
     * @param f файл.
     * @return документ, null: ошибка загрузки.
     */
    public static Document LoadXML(final File f) {
        if ( f.exists() && (f.length() > 0)) {
            //файл найден
            try (InputStream fis = new FileInputStream(f)) {
                return DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(fis);
            } catch (Exception e) {
                if (BuildConfig.DEBUG) e.printStackTrace();
            }
        }
        return null;
    }

    /**
     * Вывод в отладочный лог байтового массива.
     * @param tag название строки/префикса отладки.
     * @param data массив.
     * @param from первая позиция.
     * @param to последняя позиция.
     */
    public static void LogBytes(String tag, byte[] data, int from, int to) {
        StringBuilder deb = new StringBuilder();
        StringBuilder debT = new StringBuilder();

        if ( BuildConfig.DEBUG ) Log.d(tag, "data length="+data.length+" from="+from+" to="+to+"\n" );
        for (int i = 0; i < to - from; i++) {
            if (((i % 16) == 0) && (i != 0)) {
                Log.d(tag,  String.format("%04X:", (i / 16) - 1) + deb + "  " + debT + "\n" );
                deb = new StringBuilder();
                debT = new StringBuilder();
            }

            deb.append(String.format(" %02X", data[from + i]));

            try {
                debT.append(((0xFF & (int) data[from + i]) < 0x20) ? "." : new String(data, from + i, 1, "Cp866"));
            }
            catch(Exception ignored) {}
        }
        if ( deb.length() > 0 ) {
            while(deb.length()<48) deb.append(" ");

            if ( BuildConfig.DEBUG ) Log.d(tag,  String.format("%04X:", ((to - from) - 1)/ 16) + deb + "  " + debT + "\n" );
        }
    }

    /**
     * Добавление тегов в реквизит.
     * @param s строка к которой добавляется тег.
     * @param tagNum номер тега.
     * @param tagData данные тега.
     * @return строка с тегом.
     */
    public static String addTag(String s, int tagNum, String tagData) {
        if ( (tagData != null) && !tagData.isEmpty() )
            return "<" + tagNum + ">" + tagData + (s.isEmpty() ? "":"|"+s);
        return s;
    }

    /** Перечень типов таймаутов обращения к КФН {@link #tmGetTimeOut(tmTimeoutType)}. */
    public enum tmTimeoutType{
        /** Таймаут запроса статуса КФН. */
        TM_TIMEOUT_STATUS,
        /** Таймаут обращения к КФН. */
        TM_TIMEOUT_CMD,
        /** Таймаут команды КФН с печатью документа. */
        TM_TIMEOUT_PRINT,
        /** Таймаут команды КФН с печатью отчета. */
        TM_TIMEOUT_PRINT_REPORT,
        /** Таймаут команды КФН для работы с сетью. */
        TM_TIMEOUT_NETWORK
    }

    /**
     * Получение таймаута заданного типа для КФН.
     * @param tt тип таймаута {@link tmTimeoutType}.
     * @return значение в миллисекундах.
     */
    public static long tmGetTimeOut(tmTimeoutType tt) {
        switch (tt) {
            // Таймаут запроса статуса КФН (в миллисекундах).
            case TM_TIMEOUT_STATUS: return TMDefaults.tm_timeouts.DEFAULT_TM_TIMEOUT_STATUS;
            // Таймаут команды КФН с печатью документа (в миллисекундах).
            case TM_TIMEOUT_PRINT: return TMDefaults.tm_timeouts.DEFAULT_TM_TIMEOUT_PRINT;
            // Таймаут по команды КФН с печатью отчета (в миллисекундах)
            case TM_TIMEOUT_PRINT_REPORT: return TMDefaults.tm_timeouts.DEFAULT_TM_TIMEOUT_PRINT_REPORT;
            // Таймаут по умолчанию команды КФН для работы с сетью (в миллисекундах).
            case TM_TIMEOUT_NETWORK: return TMDefaults.tm_timeouts.DEFAULT_TM_TIMEOUT_NETWORK;
            // Таймаут обращения к КФН (в миллисекундах).
            default: return TMDefaults.tm_timeouts.DEFAULT_TM_TIMEOUT_CMD;
        }
    }
}
